/***************************************************************************
    slide.h  -  Repraesentiert ein Dia in einer Praesentation
    -------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.11 $
    CVS-ID    : $Id: slide.h,v 1.11 2002/11/10 11:35:47 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef SLIDE_H
#define SLIDE_H


// Forward-Deklarationen:
class QString;


/**
 * Slide
 *
 * Repraesentiert ein Dia. Ein Objekt der Klasse Slide besteht aus folgenden
 * Attributen:
 * <ul>
 *   <li><tt>duration</tt>: Zeitdauer (msec.) fuer die Darstellung dieses
 *                         Dias.</li>
 *   <li><tt>absTime</tt>: Absolute Zeit (msec.) seit Beginn der
 *                         Praesentation.</li>
 *   <li><tt>blendingTimeSign</tt>: Vorzeichen (<tt>none</tt>, <tt>plus</tt>
 *                         oder <tt>minus</tt>) der Ueberblendungszeit. Dabei
 *                         entspricht <tt>none</tt> <b>nicht</b>
 *                         <tt>plus</tt> !</li>
 *   <li><tt>blendingTime</tt>: Zeit (msec.) fuer die Ueberblendung.</li>
 *   <li><tt>picFilename</tt>: Dateiname des Dias incl. Pfad.</li>
 * </ul>
 * Die Angaben <tt>absTime</tt>, <tt>blendingTimeSign</tt>,
 * <tt>blendingTime</tt> und <tt>picFilename</tt> sind aus der Ablaufdatei
 * der Dia-Show zu entnehmen. Die Zeitdauer fuer dieses Dia ergibt sich aus
 * der Differenz der <tt>absTime</tt> des Nachfolgedias und der
 * <tt>absTime</tt> dieses Dias. Das <em>Dia</em> mit dem <tt>picFilename</tt>
 * "ende" symbolisiert das Ende des Ablaufplanes.
 * @see Schedule
 *
 * @author Dirk Rosert <dirk@rosert.de>
 * $Revision: 1.11 $
 */
class Slide
{
public: // types:

  /** Typ fuer das Vorzeichen der Ueberblendungszeit. */
  enum BlendingTimeSign { none  /** Kein Anpassung noetig */ =0,
                          plus  /** Bei Ueberblendung wird diese zur Zeit
                                    <tt>abstime + blendingTime/2</tt>
                                    gestartet. */ ,
                          minus /** Bei Ueberblendung wird diese zur Zeit
                                    <tt>abstime - blendingTime/2</tt>
                                    gestartet. */
                        };


public: // methods:

  /**
   * Erzeugt ein Slide mit der Darstellungsdauer @p duration (in msec.),
   * der absoluten Startzeit des Dias @p abstime (in msec), das Vorzeichen
   * der Ueberblendung @p blendingTimeSign, der Ueberblendungszeit
   * @p blendingTime und des Dateinamen (samt Pfad) @p picfilename. Wenn
   * @p duration nicht mit uebergeben wird, so wird 0 als Default angenommen.
   * Dieser Wert kann durch die Methode <tt>duration(int)</tt> spaeter
   * gesetzt werden.
   */
  Slide(int abstime, int dummy, const QString& picfilename, int duration=0,
        Slide::BlendingTimeSign sign=none);

  /**
   * Gibt die Zeitdauer (in msec.) fuer die Darstellung dieses Dias zurueck.
   */
  int duration() const;

  /**
   * Gibt die absolute Zeit (in msec.) seit Beginn der Praesentation fuer die
   * Startzeit dieses Dias zurueck.
   */
  int absTime() const;

  /**
   * Gibt das Vorzeichen der Ueberblendungszeit zurueck.
   * @see Slide#BlendingTimeSign
   */
  BlendingTimeSign blendingTimeSign() const;

  /**
   * Gibt die Zeit fuer die Ueberblendungszeit (in msec.) zum naechsten Dia
   * zurueck.
   */
  int blendingTime() const;

  /**
   * Gibt die (absolute) Startzeit fuer die Ueberblendung dieses Dias zurueck.
   * Ist eine Ueberblendung zwischen Dias moeglich / eingeschaltet, dann
   * beginnt die Ueberblendung in Abhaengigkeit vom (moeglichen) Vorzeichen
   * der Ueberblendungszeit und der Ueberblendungszeit an sich.
   */
  int startTimeOfSlideChange() const;

  /**
   * Gibt die absolute Zeit zurueck, die das Ende der Ueberblendung
   * terminiert. Sie ergibt sich aus <tt>startOfBlending() + duration</tt>.
   */
  int endTimeOfSlideChange() const;

  /**
   * Berechnet die Verschiebung der Diawechselzeit, die sich ohne
   * Ueberblendung ggf. aus dem Vorzeichen der Ueberblendungszeit ergibt.
   * <p>
   * Als Beispiel hier ein fiktiver Ablaufplan:
   * <pre>
   * +-------------------+
   * |10000   4000  bild1|
   * |20000  -4000  bild2|
   * |30000  +4000  bild3|
   * +-------------------+
   * </pre>
   * Daraus ergibt sich fuer <tt>bild1</tt>
   * <pre>
   *            #
   * --+--******#******--+--
   *   |  *  |  #  |  *  |
   *   |  *  |  #  |  *  |
   * --+--******#******--+--
   *   17 18 19 20 21 22 23
   * </pre>
   * ein Offset von <tt>0</tt>, fuer <tt>bild2</tt>
   * <pre>
   *      #
   * --+--#######******--+--
   *   |  *  |  #  |  *  |
   *   |  *  |  #  |  *  |
   * --+--******#******--+--
   *   27 28 29 30 31 32 33
   * </pre>
   * ein Offset von <tt>-4000/2</tt> und fuer <tt>bild3</tt>
   * <pre>
   *                  #
   * --+--******#######--+--
   *   |  *  |  #  |  *  |
   *   |  *  |  #  |  *  |
   * --+--******#******--+--
   *   37 38 39 40 41 42 43
   * </pre>
   * ein Offset von <tt>+4000/2</tt>.
   *
   * Der horizontale <em>Zeitstrahl</em> aus <tt>+</tt> und <tt>-</tt>
   * symbolisiert den Ablaufplan mit den Sekundenangaben unterhalb des
   * <em>Zeitstrahles</em>. Das Rechteck, umrandet der <tt>*</tt>'s zeigt den
   * Zeitbereich der Ueberblendung vom aktuellen zum folgenden Bild. Die
   * <em>Breite</em> des Rechteckes enspricht dem 'Betrag' der
   * Ueberblendungszeit im Ablaufplan (fuer jedes Bild hier 4000 <=> 4 sec).
   * Die <tt>#</tt>'s zeigen im unteren Bereich des <em>Zeitstrahles</em>
   * die Diawechselzeit aus dem Ablaufplan. Im oberen Bereich wird die
   * Verschiebung dargestellt, die sich aus dem Vorzeichen der
   * Ueberblendungszeit ergibt.
   */
  int slideChangeOffset() const;
  
  /** Gibt den Namen der Grafikdatei dieses Dias mit Pfad zurueck. */
  QString picFilename() const;

  /** Setzt die Abspieldauer des Dias. */
  void duration(int);

  /** Setzt das Vorzeichen der Ueberblendungszeit. */
  void blendingTimeSign(BlendingTimeSign);

  /** Setzt den Ueberblendungszeit des Dias. */
  void blendingTime(int);


protected: // attribtues:

  /**
   * Absolute Zeit seit Praesentationsbeginn, nach der dieses Dia dargestellt
   * werden soll.
   */
  int m_abstime;

  /** Zeit fuer die Ueberblendung. */
  int m_blendingTime;

  /** Name der zugehoerigen Grafikdatei (incl. Pfad). */
  QString m_picfilename;

  /** Zeitdauer, in der dieses Dia laut Ablaufplan angezeigt werden soll. */
  int m_duration;

  /** Vorzeichen der Ueberblendungszeit. */
  BlendingTimeSign m_blendingTimeSign;

};


#endif // SLIDE_H
