/***************************************************************************
    schedule.cpp  -  Ablaufplan einer Praesentation
    ------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.28 $
    CVS-ID    : $Id: schedule.cpp,v 1.28 2003/01/13 19:11:56 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt include files:
#include <qfile.h>
#include <qstring.h>

// Application include files:
#include "globals.h"
#include "schedule.h"
#include "slide.h"
#include "properties.h"


// erzeugt in main.cpp:
extern Properties *properties;


Schedule::Schedule(const QString& resolution,
                   const QString& schedulefilename) :
  m_resolutionStr(resolution),
  m_scheduleFilenameStr(schedulefilename)
{
  QString base = properties->base();

  m_filename = base + m_scheduleFilenameStr;

  QFile file(m_filename);
  if ( !file.open(IO_ReadOnly) )
    throw CanNotOpenFile(m_filename);

  QTextStream tstream(&file);
  if ( tstream.eof() )
    throw ContentError(m_filename);

  // Eine aufloesungsabhaengige Ablaufplan-Datei (z.B. fein.ply oder grob.ply)
  // ist folgendermassen aufgebaut:
  //
  // +---------------------------------------------------------------------+
  // | 1: "Name der Sound-Datei"                                           |
  // | 2: "abs. Zeit in msec. fuer Dia-Start" "Ueberbl." "Grafikdateiname" |
  // | 3: "abs. Zeit in msec. fuer Dia-Start" "Ueberbl." "Grafikdateiname" |
  // |    ...                                                              |
  // | n: "abs. Zeit in msec. fuer Ende der Praes." "Ueberbl." <<<ende>>>  |
  // +---------------------------------------------------------------------+
  //
  // Die doppelten Anfuehrungszeichen sind nicht Teil der Zeichenketten
  // in der Datei, sie sollen die Grenzen der einzelnen Strings hier
  // markieren. Auch die Zeilennummern sind nicht Teil der Datei.
  // <<<ende>>> steht fuer die String-Konstante ende, die als solche das
  // Ende der Praesentation anzeigt.
  //
  // Neu: die Ueberblendzeit kann auch ein Vorzeichen haben. Dieses hat
  // Einfluss auf den Diawechsel, wenn eine Ueberblendung zwischen den
  // Dias moeglich ist.
  //
  // Am Ende dieser Datei zeigt ein Beispiel, wie aus Ablaufplan-Datei
  // Ein Ablaufplan der Klasse Schedule wird.

  // === erste Zeile enthaelt den Namen der Sound-Datei ===
  if ( !tstream.eof() )
  {
    m_soundfilename = base + tstream.readLine();
    Globals::backslashToSlash(m_soundfilename);
  }
  else
  {
    throw ContentError(m_filename);
  } // END if


  // === uebrige Zeilen repreasentieren die Dias ===

  QString line, absTimeStr, blendTimeStr, filenameStr, signStr;
  int pos1, pos2, abstime, blendTime;
  Slide::BlendingTimeSign sign;
  while ( !tstream.eof() )
  {
    line = tstream.readLine();

    if ( !line.isNull() )
    {
      /*
       *   88888 99999 foobar
       *        ^     ^
       *      pos1   pos2
       */
      pos1 = line.find(" ");
      pos2 = line.findRev(" ");

      absTimeStr   = line.mid(0, pos1);
      blendTimeStr = line.mid(pos1+1, pos2);
      filenameStr  = line.mid(pos2+1, line.length());

      absTimeStr.stripWhiteSpace();
      blendTimeStr.stripWhiteSpace();
      filenameStr.stripWhiteSpace();

      // konvertiere backslash (DOS/Win) in slash (Unix):
      Globals::backslashToSlash(filenameStr);

      sscanf(absTimeStr.data(),   "%i", &abstime);
      sscanf(blendTimeStr.data(), "%i", &blendTime);

      signStr = blendTimeStr[1];
      if ( signStr == "+" )
      {
        sign = Slide::plus;
      }
      else
      {
        if ( signStr == "-" )
        {
          sign = Slide::minus;
        }
        else
          sign = Slide::none;

      } // END if

      if ( filenameStr != "ende")
        filenameStr = base + filenameStr;

      m_slides.push_back(Slide(abstime, abs(blendTime), filenameStr, sign));

    } // END if !eof()

  } // END while

  file.close();

  // erste Zeile im Ablaufplan zeigt auf Default-Grafik (siehe Ende der Datei)
  m_slides.insert(m_slides.begin(),
                  Slide(0, 0, base + properties->defaultImageFilename()));


  // === Fuelle die 'Diadarstellungszeiten' ===
  for ( unsigned int i=1; i<m_slides.size(); i++ )
  {
    m_slides[i-1].duration( m_slides[i].absTime() - m_slides[i-1].absTime() );
  } // END for

#ifndef NDEBUG
//  out();
#endif


  // === Fuelle die 'Menge der Tage' ===

  DayInterval day;

  // erster 'Tag' ist die Default-Grafik
  day.firstSlide = 0;

  // ein neuer Tag beginnt mit einem Dia, dessen Dateiname die Zeichenkette
  // '00.' enthaelt (z.B. bild1200.jpg)
  for ( unsigned int idx = 1; idx < m_slides.size(); idx++ )
  {
    if ( (m_slides[idx].picFilename()).find("00.") != -1 )
    {
      // es hat schon einen Tag gegeben, der ein Dia vorher zuende ging
      day.lastSlide = idx-1;
      m_days.push_back(day);

      // ein neuer Tag beginnt nun
      day.firstSlide = idx;

    } // END if

  } // END for

  if ( day.firstSlide > day.lastSlide )
  {
    // der letzte Tag muss noch abgeschlossen werden
    day.lastSlide = m_slides.size()-1;
    m_days.push_back(day);
  } // END if

  // --- erstes Dia als aktuelles Dia ---
  m_slideIdx = 0;

} // END Schedule()


Schedule::~Schedule()
{
  while ( m_slides.size() > 0 )
    m_slides.erase(m_slides.begin());

} // END ~Schedule()


QString Schedule::resolutionString() const
{
  QString result = m_resolutionStr;
  
  int pos = m_scheduleFilenameStr.findRev(".");
  if ( pos != -1 )
  {
    result = result + " (" + m_scheduleFilenameStr.left(pos) + ")";
  } // END if
  
  return result;
} // END resolutionString()


const QString& Schedule::scheduleFilename() const
{
  return m_filename;
} // END scheduleFilename()


const QString& Schedule::soundFilename() const
{
  return m_soundfilename;
} // END soundFilename()


unsigned int Schedule::size() const
{
  return m_slides.size();
} // END size()


int Schedule::currSlideIdx() const
{
  return m_slideIdx;
} // END currSlideIdx()


int Schedule::duration() const
{
  return m_slides[m_slides.size()-1].absTime();
} // END duration()


Slide& Schedule::slide(unsigned int idx)
{
  if ( idx >= m_slides.size() )
    throw OutOfRange();

  return m_slides[idx];
} // END slide()


unsigned int Schedule::idx() const
{
  return m_slideIdx;
} // END idx()


Slide& Schedule::slide()
{
  return m_slides[m_slideIdx];
} // END slide()


bool Schedule::nextSlide(Slide& nextslide)
{
  if ( m_slideIdx < m_slides.size() )
  {
    nextslide = m_slides[m_slideIdx+1];

    return true;
  } // END if

  return false;
} // END nextSlide()


void Schedule::prevDay(unsigned int picIdx, unsigned int& newPicIdx)
{
  unsigned int dayIdx = isPartOfDay(picIdx);

  if ( picIdx == m_days[dayIdx].firstSlide )
  {
    // picIdx ist erstes Dia des Tages, gehe zum vorigen Tag, wenn dieses
    // nicht der erste Tag ist (dann bleibe in diesem Tag)
    if ( dayIdx > 0 )
      newPicIdx = m_days[dayIdx-1].firstSlide;
    else
      newPicIdx = m_days[dayIdx].firstSlide;
  }
  else
  {
    // picIdx ist mitten im Tag, gehe zum Anfang des Tages
      newPicIdx = m_days[dayIdx].firstSlide;
  } // END if

} // END prevDay()


bool Schedule::nextDay(unsigned int picIdx, unsigned int& newPicIdx)
{
  unsigned int dayIdx = isPartOfDay(picIdx);

  // gehoert picIdx schon zum letzten Tag ?
  if ( dayIdx == m_days.size()-1 )
    return false;

  // sonst nimm erstes Dia vom naechsten Tag
  newPicIdx = m_days[dayIdx+1].firstSlide;

  return true;
} // END nextDay()


void Schedule::reset()
{
  m_slideIdx = 0;
} // END reset()


unsigned int Schedule::isPartOfDay(unsigned int slideIdx) const
{
  DayInterval day;

  for ( unsigned int dayIdx = 0; dayIdx < m_days.size(); dayIdx++ )
  {
    day = m_days[dayIdx];

    if ( (day.firstSlide <= slideIdx) && (slideIdx <= day.lastSlide) )
    {
#ifndef NDEBUG
      cerr << "Schedule::isPartOfDay(); Dia " << slideIdx
           << " geheort zum Tag Nr. " << dayIdx << endl;
#endif

      return dayIdx;
    } // END if

  } // END for

  // bis hier darf man nicht kommen !
  return 0;

} // END isPartOfDay()


#ifndef NDEBUG
void Schedule::out()
{
  // Ablaufplan soll formatiert ausgegeben werden,

  unsigned int absTimeLen=0, durationLen=0, blendTimeLen=0, filenameLen=0;
  int absTime, duration, blendTime;
  QString absTimeStr, durationStr, blendTimeStr, filename;

  // Analyse:
  for ( unsigned int i=0; i<m_slides.size(); i++ )
  {
    absTime   = m_slides[i].absTime();
    duration  = m_slides[i].duration();
    blendTime = m_slides[i].blendingTime();
    filename  = m_slides[i].picFilename();

    absTimeStr.sprintf("%i", absTime);
    durationStr.sprintf("%i", duration);
    blendTimeStr.sprintf("%i", blendTime);

    if ( absTimeStr.length() > absTimeLen )
      absTimeLen = absTimeStr.length();
    if ( durationStr.length() > durationLen )
      durationLen = durationStr.length();
    if ( blendTimeStr.length() > blendTimeLen )
      blendTimeLen = blendTimeStr.length();
    if ( filename.length() > filenameLen )
      filenameLen = filename.length();
  } // END for

  // Ausgabe:
  cout << m_scheduleFilenameStr << endl;
  Slide::BlendingTimeSign sign;
  QString signStr;
  QString absTimeHead =   "absTime";
  QString durationHead =  "duration";
  QString blendTimeHead = "blendTime";
  QString filenameHead =  "filename";

  if ( absTimeHead.length() > (absTimeLen+2) )
    absTimeHead = absTimeHead.left(absTimeLen+2);
  else
    absTimeHead = absTimeHead +
                  xChar((absTimeLen+2)-absTimeHead.length(), ' ');

  if ( durationHead.length() > (durationLen+2) )
    durationHead = durationHead.left(durationLen+2);
  else
    durationHead = durationHead +
                   xChar((durationLen+2)-durationHead.length(), ' ');

  if ( blendTimeHead.length() > (blendTimeLen+2) )
    blendTimeHead = blendTimeHead.left(blendTimeLen+2);
  else
    blendTimeHead =+ xChar((blendTimeLen+2)-blendTimeHead.length(), ' ');

  if ( filenameHead.length() > (filenameLen+2) )
    filenameHead = filenameHead.left(filenameLen+2);
  else
    filenameHead = filenameHead +
                   xChar((filenameLen+2)-filenameHead.length(), ' ');

  cout <<                               "+"
       << xChar(absTimeLen+2,   '-') << "+"
       << xChar(durationLen+2,  '-') << "+"
       << xChar(3,              '-') << "+"
       << xChar(blendTimeLen+2, '-') << "+"
       << xChar(filenameLen+2,  '-') << "+"
       << endl
       <<                  "|"
       << absTimeHead   << "|"
       << durationHead  << "|"
       << "   "         << "|"
       << blendTimeHead << "|"
       << filenameHead  << "|"
       << endl
       <<                               "+"
       << xChar(absTimeLen+2,   '-') << "+"
       << xChar(durationLen+2,  '-') << "+"
       << xChar(3,              '-') << "+"
       << xChar(blendTimeLen+2, '-') << "+"
       << xChar(filenameLen+2,  '-') << "+"
       << endl;

  unsigned int absTimeLenDiff, durationLenDiff, blendTimeLenDiff;
  unsigned int filenameLenDiff;
  for ( unsigned int i=0; i<m_slides.size(); i++ )
  {
    absTime   = m_slides[i].absTime();
    duration  = m_slides[i].duration();
    blendTime = m_slides[i].blendingTime();
    filename  = m_slides[i].picFilename();
    sign      = m_slides[i].blendingTimeSign();

    switch ( sign )
    {
    case Slide::plus:
      signStr = "+";
      break;
    case Slide::minus:
      signStr = "-";
      break;
    default:
      signStr = " ";
    } // END switch

    absTimeStr.sprintf("%i", absTime);
    durationStr.sprintf("%i", duration);
    blendTimeStr.sprintf("%i", blendTime);
    absTimeLenDiff = absTimeLen - absTimeStr.length();
    durationLenDiff = durationLen - durationStr.length();
    blendTimeLenDiff = blendTimeLen - blendTimeStr.length();
    filenameLenDiff = filenameLen - filename.length();
    cout << "| "
         << xChar(absTimeLenDiff, ' ') << absTimeStr << " | "
         << xChar(durationLenDiff, ' ') << durationStr << " | "
         << signStr << " | "
         << xChar(blendTimeLenDiff, ' ') << blendTimeStr << " | "
         << filename << xChar(filenameLenDiff, ' ') << " |" << endl;
  } // END for
  cout << "+"
       << xChar(absTimeLen+2,   '-') << "+"
       << xChar(durationLen+2,  '-') << "+"
       << xChar(3,              '-') << "+"
       << xChar(blendTimeLen+2, '-') << "+"
       << xChar(filenameLen+2,  '-') << "+"
       << endl << endl;

} // END out()


QString Schedule::xChar(unsigned int x, char c)
{
  QString result = "";
  if ( x == 0 )
    return result;

  for ( unsigned int i=0; i<x; i++ )
    result = result + c;

  return result;
} // END xChar()
#endif
