/***************************************************************************
    ktourimage.cpp  -  Repraesentiert ein ImlibImage
    --------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.15 $
    CVS-ID    : $Id: ktourimage.cpp,v 1.15 2002/12/17 19:19:43 dirk Exp $

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * basiert auf KuickShow 0.8.2 von Carsten Pfeiffer <pfeiffer@kde.org>
 */


// C include files: 
#include <sys/time.h>
#include <unistd.h>

// STL include files:
#include <iostream>

// Qt include files:
#include <qcolor.h>
#include <qevent.h>
#include <qfile.h>
#include <qlist.h>
#include <qpalette.h>
#include <qtimer.h>
#include <qwidget.h>

// KDE include files:
#include <kcursor.h>

// Application include files:
#include "ktourimage.h"
#ifndef NDEBUG
#include "globals.h"
#endif


KTourImage::KTourImage(const QString& filename, ImlibImage *im,
                       ImlibData *id, bool paintFlag) :
  QObject(0L, 0L)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourImage::KTourImage()" << endl;
#endif

  m_filename = filename;
  m_im       = im;
  m_id       = id;
  m_pixmap   = 0L;
  m_width    = im->rgb_width;
  m_height   = im->rgb_height;
  m_isDirty  = paintFlag;

  m_origWidth  = m_width;
  m_origHeight = m_height;
} // END KTourImage()


KTourImage::~KTourImage()
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourImage::~KTourImage()" << endl;
#endif

  Imlib_free_pixmap(m_id, m_pixmap);
  Imlib_destroy_image(m_id, m_im);
} // END ~KTourImage()


int KTourImage::width() const
{
  return m_width;
} // END width()


int KTourImage::height() const
{
  return m_height;
} // END height()


int KTourImage::originalWidth() const
{
  return m_origWidth;
} // END originalWidth()


int KTourImage::originalHeight() const
{
  return m_origHeight;
} // END originalHeight()


void KTourImage::resize(int width, int height)
{
  if ( (m_width == width) && (m_height == height) )
    return;

  m_width   = width;
  m_height  = height;
  m_isDirty = true;
} // END resize()


void KTourImage::restoreOriginalSize()
{
  if (m_width == m_origWidth && m_height == m_origHeight)
    return;

  m_width   = m_origWidth;
  m_height  = m_origHeight;
  m_isDirty = true;
} // END restoreOriginalSize()


ImlibImage *KTourImage::imlibImage() const
{
  return m_im;
} // END imlibImage()


ImlibData *KTourImage::imlibData() const
{
  return m_id;
} // END imlibData()


Pixmap& KTourImage::pixmap()
{
  if ( m_isDirty )
    renderPixmap();

  return m_pixmap;
} // END pixmap()


void KTourImage::renderPixmap()
{
  if ( !m_isDirty )
    return;

  if ( m_pixmap )
    Imlib_free_pixmap(m_id, m_pixmap);

  emit startRendering();

#ifndef NDEBUG
  struct timeval tms1, tms2;
  gettimeofday(&tms1, NULL);
#endif

  Imlib_render(m_id, m_im, m_width, m_height);
  m_pixmap = Imlib_move_image(m_id, m_im);

#ifndef NDEBUG
  gettimeofday(&tms2, NULL);
  int diffTime = (tms2.tv_usec - tms1.tv_usec) / 1000;
  diffTime = (diffTime < 0) ? (diffTime + 1000) : diffTime;
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourImage::renderPixmap(\"" << m_filename.latin1() << "\") ["
       << m_width << "x" << m_height
       << "] took "<< diffTime << " ms" << endl;
#endif

  emit stoppedRendering();

  m_isDirty = false;
} // END renderPixmap()


void KTourImage::setDirty(bool d)
{
  m_isDirty = d;
} // END setDirty()


bool KTourImage::isDirty() const
{
  return m_isDirty;
} // END isDirty()


const QString& KTourImage::filename() const
{
  return m_filename;
} // END filename()
