/***************************************************************************
    imagewidget.h  -  Bildet ein Imlib-Image auf ein Qt Widget ab
    -------------
    copyright : (C) 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.19 $
    CVS-ID    : $Id: imagewidget.h,v 1.19 2002/12/17 19:26:35 dirk Exp $

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef IMAGEWIDGET_H
#define IMAGEWIDGET_H


// Qt include files:
#include <qvariant.h>
#include <qevent.h>
#include <qlist.h>

// include those AFTER Qt-includes!
#include <Imlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

// Application include files:
#include "ktourimage.h"
#include "imagecache.h"

// Forward-Deklarationen:
class QTimer;
class QWidget;
class KAccel;
class KURL;
class ImageFullscreenWidget;


/**
 * <b>Klasse ImageWidget</b>
 *
 * Bildet ein Xlib-Fenster mit Imlib-Inhalt auf ein QWidget ab.
 * @see KTourImage
 * @see ImageCache
 *
 * @author Dirk Rosert <dirk@rosert.de>
 * $Revision: 1.19 $
 */
class ImageWidget : public QWidget
{
  Q_OBJECT

public:

  /**
   * Erzeugt ein ImlibWidget. Zusaetzlich wird ein passendes
   * ImageFullscreenWidget erzeugt.
   */
  ImageWidget(QWidget *parent=0, const char *name=0, WFlags f=0);

  /** Zerstoert das ImlibWidget. */
  virtual ~ImageWidget();

  /** Lade eine Grafik (in den Cache). */
  virtual bool loadImage(const QString& filename, bool paintFlag=false);

  /** Gibt den Namen dieser Imlib-Grafikdatei zurueck. */
  const QString& filename() const;

  /**
   * Laedt die Grafik mit dem Dateinamen @p filename in den Cache.
   * @return <tt>true</tt>, wenn Operation geglueckt
   */
  bool cacheImage(const QString& filename);

  /** Loescht das Image im Widget. */
  void eraseImage();

  /** Schaltet den Vollbildmodus ein oder wieder aus. */
  void setFullscreenEnabled(bool enable);

  /** Setzt den Helligkeitswert dieser Grafik ( -256..256). */
  void setBrightness(int);

  /** Setzt den Kontrastwert dieser Grafik (-256..256). */
  void setContrast(int);

  /** Setzt den Gammawert dieser Grafik (-256..256). */
  void setGamma(int);

  /** Fuege Helligkeit zum Bild hinzu. */
  void addBrightness(int);

  /** Fuege Kontrast zum Bild hinzu. */
  void addContrast(int);

  /** Fuege Gamma-Wert zum Bild hinzu. */
  void addGamma(int);

  /** Gibt den aktuellen Helligkeitswert dieser Grafik zurueck. */
  int brightness() const;

  /** Gibt den aktuellen Kontrastwert dieser Grafik zurueck. */
  int contrast() const;

  /** Gibt den aktuellen Gammawert dieser Grafik zurueck. */
  int gamma() const;

  /** Gibt die Breite des (skalierten) Bildes zurueck. */
  int imageWidth() const;

  /** Gibt die Hoehe des (skalierten) Bildes zurueck. */
  int imageHeight() const;

  /** */
  void setAutoRender(bool enable);

  /** */
  bool isAutoRenderEnabled() const;

  /** */
  void setMaxImageCache(int);

  /** Gibt die maximale Groesse des Caches zurueck. */
  int maxImageCache() const;

  /**
   * Gibt die Hintergrundfarbe dieses Widgets zurueck.
   * @see slotSetBackgroundColor
   */
  const QColor& backgroundColor() const;

  /** Liefert den Zeiger auf die <tt>ImlibData</tt>-Variable zurueck. */
  ImlibData *getImlibData() const  { return id; }

  /** */
  KTourImage *getKTourImage() const { return ktimg; }

  /** Schaltet 'Auto Hide' fuer den Mauszeiger ein/aus. */
  void setAutoHideEnabled(bool enable);

  /** Wird aufgerufen, wenn das Widget vergroessert wird. */
  virtual void resizeEvent(QResizeEvent *);

  /** Reimplementiert QWidget#mousePressEvent(). */
  virtual void mousePressEvent(QMouseEvent *e);

  /** */
  virtual void dragEnterEvent(QDragEnterEvent *e);

  /**
   * Wird aufgerufen, wenn ein Drop-Event auftritt. Durch Drag&Drop einer
   * play.ini-Datei soll diese geladen werden.
   */
  virtual void dropEvent(QDropEvent *);


public slots:

  /** Setzt die Hintergrundfarbe des Widgets. */
  void slotSetBackgroundColor(const QColor& color);

  /** */
  void updateImage();

  /** Erhoehe die Helligkeit. */
  void slotMoreBrightness();

  /** Vermindere die Helligkeit. */
  void slotLessBrightness();

  /** Erhoehe den Kontrast. */
  void slotMoreContrast();

  /** Vermindere den Kontrast. */
  void slotLessContrast();

  /** Erhoehe den Gamma-Wert. */
  void slotMoreGamma();

  /** Vermindere den Gamma-Wert. */
  void slotLessGamma();


signals:

  /**
   * Signalisiert, dass ein Fehler bei der Grafikdatei @p filename aufgetreten
   * ist.
   */
  void sigBadImage(const QString& filename);

  /**
   * Signalisiert, dass das Fenster den Fokus erhalten hat.
   */
  void sigFocusWindow(ImageWidget *);

  /**
   * Signalisiert, dass das Kontextmenue an Position @p pos angezeigt werden
   * soll.
   */
  void sigPopup(const QPoint &pos);

  /**
   * Signalisert, dass durch ein Drop-Event eine (neue) Datei geladen werden
   * soll.
   */
  void sigDropNewURL(const KURL&);


protected: // methods:

  /**
   * Erzeugt ein ImageWidget. Im Gegensatz zum oeffentlichen Konstruktor wird
   * kein ImageFullscreenWidget erzeugt, da dieser Kontruktor von dem von
   * ImageFullscreenWidget aufgerufen wird.
   */
  ImageWidget(const char* name, QWidget *parent, int wflags);

  /**
   * Veraendere die Groesse des Bildes nach @p wm @p h. Es wird aber die
   * Groesse des Arbeitsbereiches beachtet, so dass es nicht Groesser als der
   * Arbeitsbereich wird.
   */
  void resizeOptimal(int w, int h);

  /**
   * Versucht @p filename zu laden und liefert bei Erfolg einen Pointer auf
   * das <tt>KTourImage</tt>-Objekt oder <tt>0L</tt>.
   */
  KTourImage *loadImageInternal(const QString& filename,
                                bool paintFlag=false);

  /** Zeigt das Bild. */
  void showImage();

  /** Zentriert das Image. */
  void centerImage();

  /** Skaliert die Imlib-Grafik. */
  void zoomImage(float);

  /** Waendet die Modifikatoren (Helligkeit, Kontrast und Gamma) an. */
  void setImageModifier();

  /** */
  void updateWidget(bool geometryUpdate=true);

  /** */
  virtual void updateGeometry(int width, int height);

  /** */
  inline void autoUpdate(bool geometryUpdate=false)
  {
    if ( m_isAutoRendering )
      updateWidget(geometryUpdate);
  }

  /** */
  void closeEvent(QCloseEvent *);


protected: // attributes:

  /** Linke obere Ecke des Bildes. */
  int xpos, ypos;

  /** Breite und Hoehe des Widgets. */
  int m_width, m_height;

  /** Der Dateiname der Grafik. */
  QString m_filename;

  /** Zeiger auf die eigentlichen Imlib-Daten. */
  KTourImage *ktimg;

  /** Der Image-Cache. */
  ImageCache *imageCache;

  /** Zeiger auf die ImlibData-Variable. */
  ImlibData *id;

  /** Modifizierer fuer Helligkeit, Kontrast und Gamma-Wert. */
  ImlibColorModifier mod;

  /** Das Dazugehoerige ImageWidget fuer den Vollbildmodus. */
  ImageFullscreenWidget *m_fullscreen;

  /** Wird Widget im Vollbildmodus dargestellt ? */
  bool m_isFullscreen;

  /** Das Lowlevel-X-Fenster. */
  Window win;

  
protected slots:

  /**
   * Wechselt den Mauszeiger, der <em>Beschaeftigung</em> symbolisiert (z.B.
   * die Sanduhr).
   */
  void setBusyCursor();

  /** Zeige wieder den Standard-Mauszeiger. */
  void restoreCursor();


private: // methods:

  /**
   * Initialisiert das <tt>ImlibWidget</tt> und erzeugt den
   * <tt>ImageCache</tt>.
   */
  void init();

  /** */
  void initImlib();

  /** */
  QSize	maxImageSize() const;

  /** Fuehrt das Zooming auf Groesse des Widgets durch. */
  void doZooming(int, int);


private: // attributes:

  bool   m_isAutoRendering;
  int    m_maxImageCache;
  QColor m_backgroundColor;
  bool   ignore_resize_hack;

};


#endif // END IMAGEWIDGET_H
