/***************************************************************************
    imagewidget.cpp  -  Bildet ein Imlib-Image auf ein Qt Widget ab
    ---------------
    copyright : (C) 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.22 $
    CVS-ID    : $Id: imagewidget.cpp,v 1.22 2003/01/13 19:13:39 dirk Exp $

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// C include files:
#include <sys/time.h>
#include <unistd.h>

// STL include files:
#include <iostream>
#include <cstdlib>

// Qt include files:
#include <qapplication.h>
#include <qcolor.h>
#include <qdragobject.h>
#include <qfile.h>
#include <qobject.h>
#include <qpalette.h>
#include <qtimer.h>
#include <qwidget.h>

// KDE include files:
#include <kaccel.h>
#include <kapplication.h>
#include <kcursor.h>
#include <kdebug.h>
#include <kurl.h>
#include <kwin.h>

// Application include files:
#include "globals.h"
#include "imagewidget.h"
#include "imagefullscreenwidget.h"
#include "properties.h"
#include "pixbuf/pixbuf.h"

// erzeugt in main.cpp:
extern Properties *properties;

static const int ImlibOffset = 256;


ImageWidget::ImageWidget(QWidget *parent, const char *name, WFlags f) :
  QWidget(parent, name, f) ,
  m_isFullscreen(false)
{
  m_fullscreen = 0L;

  initImlib();
  init();

  // erzeuge zugehoeriges Vollbild-Image:
  m_fullscreen = new ImageFullscreenWidget(this, "fullscreen image");
  connect(m_fullscreen, SIGNAL(sigPopup(const QPoint&)),
          this,         SIGNAL(sigPopup(const QPoint&)));

  // erzeuge Cache fuer (willkuerlich) drei Images:
  imageCache = new ImageCache(id, 3);
  connect(imageCache, SIGNAL(sigBusy()), SLOT(setBusyCursor()));
  connect(imageCache, SIGNAL(sigIdle()), SLOT(restoreCursor()));

  setAcceptDrops(true);
  setAutoHideEnabled(properties->hideMouse());

  // Window Mode
  KWin::setType( winId(), NET::Normal );
  KWin::clearState( winId(), NET::StaysOnTop );
} // END ImageWidget()


// protected constructor
ImageWidget::ImageWidget(const char* name, QWidget *parent, int wflags) :
  QWidget(parent, name, wflags)
{
  initImlib();
  init();

  // *** FIXME: vvv benutze nur einen Cache fuer beides (Window/Fullscreen)
  // erzeuge Cache fuer (willkuerlich) drei Images:
  imageCache = new ImageCache(id, 3);
  connect(imageCache, SIGNAL(sigBusy()), SLOT(setBusyCursor()));
  connect(imageCache, SIGNAL(sigIdle()), SLOT(restoreCursor()));
  // *** FIXME: ^^^ benutze nur einen Cache fuer beides (Window/Fullscreen)

} // END ImageWidget()


ImageWidget::~ImageWidget()
{
  setAutoHideEnabled(false);

  delete id;

  if ( win )
    XDestroyWindow(x11Display(), win);

  if ( !m_isFullscreen )
    delete imageCache;

} // END ~ImageWidget()


bool ImageWidget::loadImage(const QString& filename, bool paintFlag)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::loadImage(" << filename
       << ", paintFlag=" << Globals::bool2qstring(paintFlag) << ")" << endl;
#endif

  if ( m_isFullscreen )
    return m_fullscreen->loadImage(filename, paintFlag);

  KTourImage *m_ktimg = loadImageInternal(filename, paintFlag);
  if ( m_ktimg )
  {
    ktimg = m_ktimg;
    m_filename = filename;

    autoUpdate(true);
    doZooming(width(), height());
    showImage();
  } // END if m_kimg

  return ( m_ktimg != 0L );
} // END loadImage()


const QString& ImageWidget::filename() const
{
  return m_filename;
} // END filename()


bool ImageWidget::cacheImage(const QString& filename)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::cacheImage(" << filename << ")" << endl;
#endif

  KTourImage *myKtimg = loadImageInternal(filename);

  return ( myKtimg != 0L );
} // END cacheImage()


void ImageWidget::eraseImage()
{
  if ( !ktimg )
    return;

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::eraseImage()" << endl;
#endif

  ktimg = 0L;

  ignore_resize_hack = false;
  xpos = 0;
  ypos = 0;

  XDestroyWindow(x11Display(), win);
  setAutoRender(true);
  slotSetBackgroundColor(m_backgroundColor);
  win = XCreateSimpleWindow(x11Display(), winId(), 0, 0, 1, 1, 0, 0, 0);

  if ( !m_isFullscreen )
  {
    if ( imageCache )
      delete imageCache;

    imageCache = new ImageCache(id, 3);
    connect(imageCache, SIGNAL(sigBusy()), SLOT(setBusyCursor()));
    connect(imageCache, SIGNAL(sigIdle()), SLOT(restoreCursor()));
  } // END if

  initImlib();

} // END eraseImage()


void ImageWidget::setAutoHideEnabled(bool flag)
{
  if ( flag )
  {
    qApp->installEventFilter(this);
    KCursor::setAutoHideCursor(this, flag);
    KCursor::setHideCursorDelay(properties->hideDelay());
    if ( m_fullscreen )
    {
      qApp->installEventFilter(m_fullscreen);
      KCursor::setAutoHideCursor(m_fullscreen, flag);
    } // END if
  }
  else
  {
    qApp->removeEventFilter(this);
    if ( m_fullscreen )
      qApp->removeEventFilter(m_fullscreen);

  } // END if

} // END setAutoHideEnabled()


void ImageWidget::addBrightness(int factor)
{
  if ( factor == 0 )
    return;

  int oldValue = mod.brightness - ImlibOffset;
  setBrightness(oldValue + (properties->brightnessFactor() * (int) factor));
} // END addBrightness()


void ImageWidget::addContrast(int factor)
{
  if ( factor == 0 )
    return;

  int oldValue = mod.contrast - ImlibOffset;
  setContrast(oldValue + (properties->contrastFactor() * (int) factor));
} // END addContrast()


void ImageWidget::addGamma(int factor)
{
  if ( factor == 0 )
    return;

  int oldValue = mod.gamma - ImlibOffset;
  setGamma(oldValue + (properties->gammaFactor() * (int) factor));
} // END addGamma()


void ImageWidget::slotMoreBrightness()
{
  addBrightness(properties->brightnessSteps());
} // END slotMoreBrightness()


void ImageWidget::slotMoreContrast()
{
  addContrast(properties->contrastSteps());
} // END slotMoreContrast()


void ImageWidget::slotMoreGamma()
{
  addGamma(properties->gammaSteps());
} // END slotMoreGamma()


void ImageWidget::slotLessBrightness()
{
  addBrightness(-properties->brightnessSteps());
} // END slotLessBrightness()


void ImageWidget::slotLessContrast()
{
  addContrast(-properties->contrastSteps());
} // END slotLessContrast()


void ImageWidget::slotLessGamma()
{
  addGamma(-properties->gammaSteps());
} // END slotLessGamma()


void ImageWidget::resizeEvent(QResizeEvent *e)
{
  if ( e->size() == e->oldSize() )
    return;

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::resizeEvent(" << e->size().width()
       << ", " << e->size().height() << ")" << endl;
#endif

  if ( ignore_resize_hack )
  {
    ignore_resize_hack = false;

    int w = width();
    int h = height();
    if ( w == QApplication::desktop()->width()  &&
         h == QApplication::desktop()->height() &&
         imageWidth() < w                       &&
         imageHeight() < h )
    {
      return;
    } // END if
  } // END if

  m_width  = e->size().width();
  m_height = e->size().height();

  doZooming(m_width, m_height);
} // END resizeEvent()


void ImageWidget::mousePressEvent(QMouseEvent *e)
{
  if ( e->button() == QMouseEvent::RightButton )
    emit sigPopup(e->globalPos());

} // END mousePressEvent()


void ImageWidget::dragEnterEvent(QDragEnterEvent *event)
{
  event->accept(QTextDrag::canDecode(event));
} // END dragEnterEvent()


void ImageWidget::dropEvent(QDropEvent *event)
{
  QString text;

  if ( QTextDrag::decode(event, text) )
  {
    // Drag Event erhalten, handelt es sich um eine play.ini-Datei ?
    QString playfile = properties->playFilename();
    if ( text.right(playfile.length()) == playfile )
      emit sigDropNewURL(text);

  } // END if

} // END dropEvent()


void ImageWidget::closeEvent(QCloseEvent *e)
{
  e->accept();

  QWidget::closeEvent(e);
} // END closeEvent()


KTourImage *ImageWidget::loadImageInternal(const QString& filename,
                                           bool paintFlag)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::loadImageInternal(" << filename << ", paintFlag="
       << Globals::bool2qstring(paintFlag) << ")" << endl;
#endif

  // weise Standardmodifikatoren dem Image zu:
  mod.brightness = ImlibOffset + properties->brightness();
  mod.contrast   = ImlibOffset + properties->contrast();
  mod.gamma      = ImlibOffset + properties->gamma();

  KTourImage *m_ktimg = imageCache->getImage(filename, mod, paintFlag);
  if ( !m_ktimg )
  {
    // Datei konnte nicht geladen werden, defekte Datei oder falsches Format
    kdDebug() << "KTourPresenter(ImageWidget): can't load image "
              << filename.latin1() << " !" << endl;
    return 0L;
  } // END if

  return m_ktimg;
} // END loadImageInternal()


void ImageWidget::setFullscreenEnabled(bool enable)
{
  m_isFullscreen = enable;

  if ( enable )
  {
    // lade Dia in das Vollbild-Widget
    loadImage(m_filename);
    m_fullscreen->show();
    setFocus();
    topLevelWidget()->setActiveWindow();
  }
  else
  {
    // lade Dia in den Anwendungsfenster
    loadImage(m_fullscreen->filename());
    m_fullscreen->hide();
    topLevelWidget()->setActiveWindow();
  } // END if

} // END setFullscreenEnabled()


void ImageWidget::showImage()
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::showImage()" << endl;
#endif

  XMapWindow(x11Display(), win);
  XSync(x11Display(), false);
} // END showImage()


void ImageWidget::zoomImage(float factor)
{
  if ( factor == 1 || factor == 0 || !ktimg )
    return;

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::zoomImage(" << factor << ")" << endl;
#endif

  float wf, hf;

  wf = (float) ktimg->width() * factor;
  hf = (float) ktimg->height() * factor;

  // minimale Groesse fuer eine Image sind 2x2 pixels
  if ( wf <= 2.0 || hf <= 2.0 )
    return;

  ktimg->resize((int) wf, (int) hf);
  autoUpdate(true);

} // END zoomImage()


// -256..256
void ImageWidget::setBrightness(int factor)
{
  if ( m_isFullscreen )
  {
    m_fullscreen->setBrightness(factor);
    return;
  } // END if

  mod.brightness = factor + ImlibOffset;
  setImageModifier();
  properties->brightness(factor);
  autoUpdate();
} // END setBrightness()


// -256..256
void ImageWidget::setContrast(int factor)
{
  if ( m_isFullscreen )
  {
    m_fullscreen->setContrast(factor);
    return;
  } // END if

  mod.contrast = factor + ImlibOffset;
  setImageModifier();
  properties->contrast(factor);
  autoUpdate();
} // END setContrast()


// -256..256
void ImageWidget::setGamma(int factor)
{
  if ( m_isFullscreen )
  {
    m_fullscreen->setGamma(factor);
    return;
  } // END if

  mod.gamma = factor + ImlibOffset;
  setImageModifier();
  properties->gamma(factor);
  autoUpdate();
} // END setGamma()


void ImageWidget::slotSetBackgroundColor(const QColor& color)
{
  if ( m_fullscreen && !m_isFullscreen )
    m_fullscreen->slotSetBackgroundColor(color);

  setBackgroundMode(PaletteBackground);
  m_backgroundColor = color;
  setPalette(QPalette(color));
  setEraseColor(color);
} // END slotBackgroundColor()


void ImageWidget::updateImage()
{
  updateWidget(true);
} // END updateImage()


void ImageWidget::updateWidget(bool geometryUpdate)
{
  if ( !ktimg )
    return;

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::updateWidget("
       << ( geometryUpdate ? "true" : "false") << ")" << endl;
#endif

  XSetWindowBackgroundPixmap(x11Display(), win, ktimg->pixmap());
  XClearWindow(x11Display(), win);

  if ( geometryUpdate )
    updateGeometry(ktimg->width(), ktimg->height());

  showImage();
} // END updateWidget()


void ImageWidget::updateGeometry(int imWidth, int imHeight)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::updateGeometry(" << imWidth << "x" << imHeight
       << ")" << endl;
#endif

  XResizeWindow(x11Display(), win, imWidth, imHeight);

  if ( imWidth != (int) m_width || imHeight != (int) m_height )
  {
    if ( m_isFullscreen )
      centerImage();
    else
      resizeOptimal(imWidth, imHeight); // also centers the image
  }
  else
  {
    // image size == widget size
    xpos = 0; ypos = 0;
    XMoveWindow( x11Display(), win, 0, 0 );
  } // END if

} // END updateGeometry()


const QColor& ImageWidget::backgroundColor() const
{
  return m_backgroundColor;
} // END backgroundColor


void ImageWidget::setImageModifier()
{
  if ( !ktimg )
    return;

  Imlib_set_image_modifier(id, ktimg->imlibImage(), &mod);
  ktimg->setDirty(true); // laesst Grafik neuzeichnen
} // END setImageModifier()


int ImageWidget::imageWidth() const
{
  return ktimg ? ktimg->width() : 0;
} // END imageWidth()


int ImageWidget::imageHeight() const
{
  return ktimg ? ktimg->height() : 0;
} // END imageHeight()


void ImageWidget::setAutoRender(bool enable)
{
  m_isAutoRendering = enable;
} // END setAutoRender()


bool ImageWidget::isAutoRenderEnabled() const
{
  return m_isAutoRendering;
} // END isAutoRenderEnabled()


int ImageWidget::maxImageCache() const
{
  return m_maxImageCache;
} // END maxImageCache()


void ImageWidget::setBusyCursor()
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::waitCursor());
} // END setBusyCursor()


void ImageWidget::restoreCursor()
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::arrowCursor());
} // END restoreCursor()


void ImageWidget::resizeOptimal(int w, int h)
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::resizeOptimal(" << w << ", " << h << ")" << endl;
#endif

  QSize s = maxImageSize();

  int mw = s.width();
  int mh = s.height();
  int neww = (w >= mw) ? mw : w;
  int newh = (h >= mh) ? mh : h;

  if ( (neww == m_width) && (newh == m_height) )
    centerImage();
  else
    resize(neww, newh); // zentriert ebenfalls das Image

} // END resizeOptimal()


void ImageWidget::centerImage()
{
  xpos = m_width/2 - imageWidth()/2;
  ypos = m_height/2 - imageHeight()/2;

  XMoveWindow(x11Display(), win, xpos, ypos);
} // END centerImage


QSize ImageWidget::maxImageSize() const
{
  QSize result;

  if ( m_isFullscreen )
    result = QApplication::desktop()->size();
  else
    result = Globals::workArea().size() - Globals::frameSize(winId());

  return result;
} // END maxImageSize()


void ImageWidget::init()
{
  int w              = 1; // > 0 fuer XCreateWindow
  int h              = 1;
  m_backgroundColor  = properties->backgroundColor();
  ktimg              = 0L;
  ignore_resize_hack = false;
  xpos = 0;
  ypos = 0;

  if ( !id )
    fatal("ImageWidget: Imlib not initialized, aborting.");

  setAutoRender(true);

  slotSetBackgroundColor(m_backgroundColor);

  win = XCreateSimpleWindow(x11Display(), winId(), 0, 0, w, h, 0, 0, 0);

} // END init()


void ImageWidget::initImlib()
{
  ImlibInitParams par;
  par.flags = ( PARAMS_REMAP |
      PARAMS_FASTRENDER | PARAMS_HIQUALITY | PARAMS_DITHER |
      PARAMS_IMAGECACHESIZE | PARAMS_PIXMAPCACHESIZE );

  unsigned int maxcache = properties->maxCache();
  par.paletteoverride   = properties->ownPalette()  ? 1 : 0;
  par.remap             = properties->fastRender()  ? 1 : 0;
  par.hiquality         = properties->dither16bit() ? 1 : 0;
  par.dither            = properties->dither8bit()  ? 1 : 0;
  par.imagecachesize    = maxcache * 1024;
  par.pixmapcachesize   = maxcache * 1024;

  id = Imlib_init_with_params(x11Display(), &par);
} // END initImlib()


void ImageWidget::doZooming(int width, int height)
{
  if ( !ktimg )
    return; // kein Image zu zoomen

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "ImageWidget::doZooming(" << width << "x" << height << ")" << endl;
#endif

  float faktorWidth =
    (float) width / (float) ktimg->width();
  float faktorHeight =
    (float) height / (float) ktimg->height();

  if ( faktorWidth < faktorHeight )
    zoomImage(faktorWidth);
  else
    zoomImage(faktorHeight);

  centerImage();
} // END doZooming()
