/***************************************************************************
    soundengine.cpp  -  Low Level Sound Engine
    ---------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.16 $
    CVS-ID    : $Id: soundengine.cpp,v 1.16 2002/12/22 15:30:25 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * basiert auf Noatun
 */


// C include files:
#include <sys/wait.h>  // for the macro WIFEXITED

// STL include files:
#include <vector>

// Qt include files:
#include <qdir.h>
#include <qfile.h>
#include <qstring.h>

// KDE/aRTs include files:
#include <arts/connect.h>
#include <kconfig.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kstddirs.h>
#include <kurl.h>

// Application include files:
#include "soundengine.h"
#ifndef NDEBUG
#include "globals.h"
#endif


/**
 * Entnommen aus Noatun
 */
class SoundEngine::SoundEnginePrivate
{
public:
  SoundEnginePrivate() :
    playobj(Arts::PlayObject::null()),
    server(Arts::SimpleSoundServer::null()),
    globalEffectStack(Arts::StereoEffectStack::null()),
    effectStack(Arts::StereoEffectStack::null()),
    volumeControl(Arts::StereoVolumeControl::null())
  {}
  
  Arts::PlayObject playobj;
  Arts::SimpleSoundServer server;
  Arts::StereoEffectStack globalEffectStack;
  Arts::StereoEffectStack effectStack;
  Arts::Synth_AMAN_PLAY amanPlay;
  
  int volumeID;
  Arts::StereoVolumeControl volumeControl;
};


// -------------------------------------------------------------------------


static QCString mimetype(const QString& file)
{
  KMimeType::Ptr mimetype = KMimeType::findByURL( KURL( file ) );

  return mimetype->name().latin1();
} // END mimetype()


static string getObjectType(const QString& file)
{
  string objectType;

  Arts::TraderQuery query;
  query.supports("Interface", "Arts::PlayObject");
  query.supports("MimeType", std::string(mimetype(file)));

  vector<Arts::TraderOffer> *offers = query.query();

  if ( !offers->empty() )
  {
    objectType = offers->front().interfaceName();
  } // END if
  
  // prefer WAVPlayObject when available (don't cache wavs)
  vector<Arts::TraderOffer>::iterator i;
  for ( i = offers->begin(); i != offers->end(); i++ )
  {
    if ( i->interfaceName() == "WAVPlayObject" )
      objectType = i->interfaceName();
  } // END for

  delete offers;
    
  return objectType;
} // END getObjectType()


static Arts::StereoVolumeControl createVolumeControl(
      Arts::SimpleSoundServer &server, Arts::StereoEffectStack &es, int &id)
{
  Arts::StereoVolumeControl result;
  
  result =
    Arts::DynamicCast(server.createObject("Arts::StereoVolumeControl"));
  result.start();
  id = es.insertBottom(result, "Volume Control");
  
  return result;
} // END createVolumeControl()


SoundEngine::SoundEngine() : QObject()
{
  dispatcher = new Arts::Dispatcher;

  d = new SoundEnginePrivate;

  // verbinde mit aRTs:
  if ( !initArts() )
  {
    KMessageBox::sorry(0,
               i18n("There was an error to connect the sound server !\n"
                    "I'm unable to play the sound."),
               i18n("Sound Server Error"));
    m_valid = false;
  }
  else
  {
    m_valid = true;
  } // END if !initArts()

} // END SoundEngine()


SoundEngine::~SoundEngine()
{
  stop();
  d->server = Arts::SimpleSoundServer::null();
  delete d;
} // END ~SoundEngine()


bool SoundEngine::open(const QString& filename)
{
  m_filename = filename;

  if ( !initArts() )
    return false;

  string objectType = getObjectType(filename);
  if ( !objectType.length() )
    return false;

  d->playobj = Arts::DynamicCast(d->server.createObject(objectType));
  if ( !d->playobj.loadMedia(filename.ascii()) )
    return false;

  if ( d->playobj.isNull() )
    return false;

  d->playobj._node()->start();

  Arts::connect(d->playobj, "left", d->globalEffectStack, "inleft");
  Arts::connect(d->playobj, "right", d->globalEffectStack, "inright");

  return true;
} // END open()


bool SoundEngine::play()
{
  if ( d->playobj.isNull() )
  {
    m_valid = false;
    
    return false;
  } // END if

  d->playobj.play();

  return true;
} // END play()


void SoundEngine::pause()
{
  if ( d->playobj.isNull() )
  {
    m_valid = false;
    
    return;
  } // END if
  
  d->playobj.pause();
} // END pause()


void SoundEngine::stop()
{
  if ( d->playobj.isNull() )
    return;

  d->playobj.halt();
  d->playobj = Arts::PlayObject::null();
} // END stop()


void SoundEngine::seek(int msec)
{
  if ( d->playobj.isNull() )
  {
    m_valid = false;
    
    return;
  } // END if

  Arts::poTime t;
  t.seconds = msec / 1000;
  t.ms      = msec % 1000;
  
  if ( !d->playobj.isNull() )
    d->playobj.seek(t);

} // END seek()


long SoundEngine::currentTime()
{
  if ( d->playobj.isNull() )
  {
    m_valid = false;

    return -1;
  } // END if

  Arts::poTime potime = d->playobj.currentTime();

  return (1000*potime.seconds) + potime.ms;
} // END if


int SoundEngine::state()
{
  if ( !d->playobj.isNull() )
    return d->playobj.state();
  else
    return Arts::posIdle;

} // END state()


Arts::SimpleSoundServer *SoundEngine::server() const
{
  return &d->server;
} // END server()


Arts::SimpleSoundServer *SoundEngine::simpleSoundServer() const
{
  return &d->server;
} // END simpleSoundServer()


Arts::PlayObject SoundEngine::playObject() const
{
  return d->playobj;
} // END playObject()


// siehe Engine::initArts() von noatun:
bool SoundEngine::initArts()
{
  if ( d->server.isNull() || d->server.error() )
  {
    d->server = Arts::Reference("global:Arts_SimpleSoundServer");

    if ( d->server.isNull() || d->server.error() )
    {
      // scheint das aRTs nicht laeuft, versuche es zu starten.
      // Erst sollte die Konfiguration von kcmarts gelesen werden.
      KConfig *config = new KConfig("kcmartsrc");
      QCString cmdline;

      config->setGroup("Arts");

      bool rt = config->readBoolEntry("StartRealTime", false);
      bool x11Comm = config->readBoolEntry("X11GlobalComm", false);

      // packe die Werte von x11Comm in .mcoprc
      KConfig *X11CommConfig = new KConfig(QDir::homeDirPath()+"/.mcoprc");

      if(x11Comm)
        X11CommConfig->writeEntry("GlobalComm", "Arts::X11GlobalComm");
      else
        X11CommConfig->writeEntry("GlobalComm", "Arts::TmpGlobalComm");

      X11CommConfig->sync();
      delete X11CommConfig;

      cmdline = QFile::encodeName(
          KStandardDirs::findExe(QString::fromLatin1("kdeinit_wrapper")));
      cmdline += " ";

      if (rt)
      {
        cmdline += QFile::encodeName(KStandardDirs::findExe(
        QString::fromLatin1("artswrapper")));
      }
      else
      {
        cmdline += QFile::encodeName(KStandardDirs::findExe(
        QString::fromLatin1("artsd")));
      } // END if
      
      cmdline += " ";
      cmdline += config->readEntry("Arguments", "-F 5 -S 8192").utf8();

      int status=system(cmdline);

      if ( status != -1 && WIFEXITED(status) )
      {
        // We could have a race-condition here.
        // The correct way to do it is to make artsd fork-and-exit
        // after starting to listen to connections (and running artsd
        // directly instead of using kdeinit), but this is better
        // than nothing.
        int time = 0;
        do
        {
          sleep(1);
          d->server = Arts::Reference("global:Arts_SimpleSoundServer");
        } while ( ++time < 5 && (d->server.isNull()) );
      
      } // END if status
      
    } // END if d->server.isNull() || d->server.error() 
  
    if ( !d->server.isNull() )
    {
      d->amanPlay = 
        Arts::DynamicCast(d->server.createObject("Arts::Synth_AMAN_PLAY"));
      d->amanPlay.title("ktourpresenter");
      d->amanPlay.autoRestoreID("ktourpresenter");
      d->amanPlay.start();
      
      d->globalEffectStack =
        Arts::DynamicCast(d->server.createObject("Arts::StereoEffectStack"));
      d->globalEffectStack.start();
      Arts::connect(d->globalEffectStack, d->amanPlay);
      
      d->effectStack =
        Arts::DynamicCast(d->server.createObject("Arts::StereoEffectStack"));
      d->effectStack.start();
      d->globalEffectStack.insertBottom(d->effectStack, "Effects Stack");
      
      d->volumeControl =
        createVolumeControl(d->server, d->globalEffectStack, d->volumeID);

    } // END if d->server.isNull()
    
  } // END if d->server.isNull() || d->server.error() 
  
  d->playobj = Arts::PlayObject::null();
  
  return true;
} // END initArts()
