/***************************************************************************
    ktourpresenterdoc.cpp  -  Dokumentenobjekt von KTourPresenter
    ---------------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.34 $
    CVS-ID    : $Id: ktourpresenterdoc.cpp,v 1.34 2003/01/18 17:59:24 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt include files:
#include <qdir.h>
#include <qfile.h>
#include <qstring.h>
#include <qstrlist.h>
#include <qtextstream.h>
#include <qwidget.h>

// KDE include files:
#include <kio/job.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>

// Application include files:
#include "globals.h"
#include "ktourpresenterdoc.h"
#include "ktourpresenter.h"
#include "ktourpresenterview.h"
#include "properties.h"
#include "schedule.h"
#include "slide.h"


QList<KTourPresenterView> *KTourPresenterDoc::pViewList = 0L;

// erzeugt in main.cpp:
extern Properties *properties;


KTourPresenterDoc::KTourPresenterDoc(QWidget *parent, const char *name) : 
  QObject(parent, name)
{
  if ( !pViewList )
    pViewList = new QList<KTourPresenterView>();

  pViewList->setAutoDelete(true);
  
  m_scheduleIdx = 0;
} // END KTourPresenterDoc()


KTourPresenterDoc::~KTourPresenterDoc()
{
} // END ~KTourPresenterDoc()


void KTourPresenterDoc::addView(KTourPresenterView *view)
{
  pViewList->append(view);
} // END addView()


void KTourPresenterDoc::removeView(KTourPresenterView *view)
{
  pViewList->remove(view);
} // END removeView()


void KTourPresenterDoc::setURL(const KURL &url)
{
  doc_url=url;
} // END setURL()


const KURL& KTourPresenterDoc::URL() const
{
  return doc_url;
} // END URL()


void KTourPresenterDoc::slotUpdateAllViews(KTourPresenterView *sender)
{
  KTourPresenterView *w;
  
  if ( pViewList )
  {
    for ( w = pViewList->first(); w != 0; w = pViewList->next() )
    {
      if ( w != sender )
        w->repaint();

    } // END for
  } // END if

} // END slotUpdateAllViews()


void KTourPresenterDoc::closeDocument()
{
  deleteContents();
} // END closeDocument()


bool KTourPresenterDoc::newDocument()
{

  return true;

} // END newDocument()


bool KTourPresenterDoc::openDocument(const KURL& url)
{
  QString tmpfile;
  KIO::NetAccess::download(url, tmpfile);

  if ( url.isValid() )
  {
    QString protocol = url.protocol();
    QString pathfile = url.path();
    
    if ( protocol != "file" )
    {
      KMessageBox::sorry(0L,
                         i18n("You can only choose a local file.\n"
                              "Remote files are not supported !"),
                         i18n("Error"));
          
      return false;
    }
    else
    {
      // lade Datei:
      if ( QFile::exists(pathfile) )
      {
        // play.ini ist folgendermassen aufgebaut:
        //
        // +--------------------------------------------------------------+
        // | 1: "Name der Praesentation"                                  |
        // | 2: "Aufloesungsstring" "Dateiname der aufloesungsabh. Datei" |
        // | 3: "Aufloesungsstring" "Dateiname der aufloesungsabh. Datei" |
        // |    ...                                                       |
        // +--------------------------------------------------------------+
        //
        // Die doppelten Anfuehrungszeichen sind nicht Teil der Zeichenketten
        // in der Datei, sollen die Grenzen der einzelnen Strings hier
        // markieren. Auch die Zeilennummern sind nicht Teil der Datei.

        QFile file(pathfile);
        if ( !file.open(IO_ReadOnly) )
        {
          emit sigCanNotOpenFile(pathfile);
          return false;
        } // END

        QTextStream tstream(&file);
        if ( tstream.eof() )
        {
          emit sigContentError();
          return false;
        } // END if

        // 1. Zeile ist Praesentationsname
        m_presentationname = tstream.readLine();

        QString line, res, schedulefilename;
        int pos;
        while ( !tstream.eof() )
        {
          line = tstream.readLine();
          line.stripWhiteSpace();

          if ( !line.isNull() )
          {
            pos              = line.find(" ");
            res              = line.mid(0, pos);
            schedulefilename = line.mid(pos+1, line.length());
            Globals::backslashToSlash(schedulefilename);

            m_schedules.push_back(Schedule(res.stripWhiteSpace(),
                                  schedulefilename.stripWhiteSpace()));
          } // END if

        } // END while

        file.close();

        m_picIdx      = 0;  // die Default-Grafik im ...
        m_scheduleIdx = 0;  // ... 1. Ablaufplan ist aktuell
      }
      else
      {
        KMessageBox::sorry(0L,
            i18n("Can't open %1. File not found !").arg(pathfile),
            i18n("File not found"));

        return false;
      } // END if exists()
    } // END if protocol != file
  }
  else
  {
    return false;
  } // END if isValid()
  
  KIO::NetAccess::removeTempFile(tmpfile);

  return true;

} // END openDocument()


void KTourPresenterDoc::deleteContents()
{
  while ( m_schedules.size() > 0 )
    m_schedules.erase(m_schedules.begin());

} // END deleteContents()


QString KTourPresenterDoc::presentationName() const
{
  return m_presentationname;
} // END presentationName()


Schedule& KTourPresenterDoc::schedule()
{
  return m_schedules[m_scheduleIdx];
} // END schedule()


Slide& KTourPresenterDoc::slide(unsigned int idx)
{
  return schedule().slide(idx);
} // END slide()


Slide& KTourPresenterDoc::slide()
{
  return m_schedules[m_scheduleIdx].slide(m_picIdx);
} // END slide()


int KTourPresenterDoc::slideIdx() const
{
  return m_picIdx;
} // END slideIdx()


const QString& KTourPresenterDoc::soundFilename() const
{
  return m_schedules[m_scheduleIdx].soundFilename();
} // END soundFilename()


void KTourPresenterDoc::next()
{
  m_picIdx++;
} // END next()


void KTourPresenterDoc::prev()
{
  m_picIdx--;
} // END prev()
  

void KTourPresenterDoc::prevDay(unsigned int& newPicIdx)
{
  m_schedules[m_scheduleIdx].prevDay(m_picIdx, newPicIdx);
} // END slotControlPrevDay()


bool KTourPresenterDoc::nextDay(unsigned int& newPicIdx)
{
  return m_schedules[m_scheduleIdx].nextDay(m_picIdx, newPicIdx);
} // END slotControlNextDay()


int KTourPresenterDoc::totalTime() const
{
  return m_schedules[m_scheduleIdx].duration();
} // END totalTime()


void KTourPresenterDoc::reset()
{
  m_picIdx = 0; // Bild 0 ist die Default-Grafik
} // END reset()


bool KTourPresenterDoc::isFirstSlide() const
{
  return ( m_picIdx < 1 );
} // END isFirstSlide()


bool KTourPresenterDoc::isLastSlide() const
{
  return ( (m_schedules[m_scheduleIdx].size() - m_picIdx) <= 1 );
} // END isLastSlide()


int KTourPresenterDoc::seek(int msec)
{
  int lowIdx  = 0;
  int highIdx = m_schedules[m_scheduleIdx].size() - 1;

  // benutze eine Art von binaerer Suche fuer das zu suchende Dia
  return seek(msec, lowIdx, highIdx);
} // END seek()


bool KTourPresenterDoc::setScheduleByName(const QString& name)
{
  int idx = 0;
  vector<Schedule>::const_iterator it;
  for ( it = m_schedules.begin(); it != m_schedules.end(); it++, idx++ )
  {
    if ( name == it->resolutionString() )
    {
      // wurde wirklich eine neue Aufloesung gewaehlt ?
      if ( m_scheduleIdx != idx )
      {
#ifndef NDEBUG
        cerr << "KTourPresenterDoc::setPresentation(" << name <<"); "
             << "new subsession idx = " << idx << endl;
#endif

        m_scheduleIdx = idx;

        return true;
      } // END if scheduleIdx

    } // END if name

  } // END for

  // dann gibt es wohl keine Aufloesung (keinen Show-Teil) mit diesem Namen !
  return false;
  
} // END setSchedukeByName()


void KTourPresenterDoc::resStrList(QStrList& res)
{
  res.clear();

  vector<Schedule>::const_iterator it;
  for ( it = m_schedules.begin(); it != m_schedules.end(); it++ )
    res.append( it->resolutionString() );

} // END resStrList()


// private:
// --------

int KTourPresenterDoc::seek(int msec, int lowIdx, int highIdx)
{
  int midIdx  = ( (highIdx - lowIdx) / 2 ) + lowIdx;
  
  int midIdxTime  = m_schedules[m_scheduleIdx].slide(midIdx).absTime();
  int midIdx2Time = m_schedules[m_scheduleIdx].slide(midIdx+1).absTime();

  if ( (midIdxTime <= msec) && (msec < midIdx2Time) )
  {
    // Dia gefunden

    m_picIdx = midIdx;
  }
  else
  {
    if ( msec < midIdxTime )
    {
      // lowIdx <= msec < midIdx

      return seek(msec, lowIdx, midIdx);
    }
    else
    {
      // midIdx < msec <= highIdx

      return seek(msec, midIdx, highIdx);
    } // END if

  } // END if
  
  return midIdxTime;
} // END seek()
