/***************************************************************************
    ktourpresenter.cpp  -  Die Hauptklasse der Anwendung
    ------------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@rosert.de
    author    : $Author: dirk $
    revision  : $Revision: 1.80 $
    CVS-ID    : $Id: ktourpresenter.cpp,v 1.80 2003/01/19 19:45:53 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// STL include files:
#include <iostream>

// Qt include files:
#include <qcstring.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qevent.h>
#include <qlabel.h>
#include <qlist.h>
#include <qstrlist.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#ifdef index
#undef index
#endif

#ifdef KeyPress
#undef KeyPress
#endif

// KDE include files:
#include <kaccel.h>
#include <kaction.h>
#include <kapplication.h>
#include <kcombobox.h>
#include <kconfig.h>
#include <kcursor.h>
#include <kdatastream.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kkeydialog.h>
#include <kled.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <krun.h>
#include <kstatusbar.h>
#include <kstdaction.h>

// Application include files:
#include "ktourpresenter.h"
#include "ktourpresenterview.h"
#include "ktourpresenterdoc.h"
#include "properties.h"
#include "configuredialog.h"
#include "schedule.h"
#include "screensavercontrol.h"
#include "slide.h"
#include "soundplayer.h"
#include "startuplogo.h"
#ifndef NDEBUG
#include "globals.h"
#endif

// Action names:
#define ACTIONPLAYPRES     "Play Presentation"
#define ACTIONSTOPPRES     "Stop Presentation"
#define ACTIONHALTPRES     "Halt Presentation"

#define ACTIONPREVDAY      "Previous Day"
#define ACTIONNEXTDAY      "Next Day"
#define ACTIONPREVSLIDE    "Previous Slide"
#define ACTIONNEXTSLIDE    "Next Slide"

#define ACTIONMORECONTR    "More Contrast"
#define ACTIONLESSCONTR    "Less Contrast"
#define ACTIONMOREBRIGHT   "More Brightness"
#define ACTIONLESSBRIGHT   "Less Brightness"
#define ACTIONMOREGAMMA    "More Gamma"
#define ACTIONLESSGAMMA    "Less Gamma"
#define ACTIONRESETMOD     "Reset Modifier"

#define ACTIONSHOWREPORT   "Show Tour Report"

#define ACTIONFULLSCREEN   "Toggle Fullscreen"


// Liste der Applikationsfenster:
QList<KTourPresenterApp> *KTourPresenterApp::windowList = 0L;

// erzeugt in main.cpp:
extern Properties *properties;


KTourPresenterApp::KTourPresenterApp(const QString& playfilename,
                                     QWidget *, const char *name) :
  KMainWindow(0L, name, WType_TopLevel | WStyle_ContextHelp) ,
  adjustOk(true)
{
  // wenn das letzte Anwendungsfenster geschlossen wurde, beende Anwendung:
  connect(kapp, SIGNAL(lastWindowClosed()),
          this, SLOT(slotLastWindowClosed()));

  // bisher noch kein Logo dargestellt:
  bool showLogo = false;

  player = 0L;
  configdialog = 0L;

  // bisher keine Praesentation:
  status = none;

  // erzeuge ggf. die Fensterliste:
  if ( !windowList )
  {
    windowList = new QList<KTourPresenterApp>;
    windowList->setAutoDelete(false);
  } // END if

  // fuege dieses Applikationsfenster ans Ende der Liste an:
  windowList->append(this);

  // ermittle Konfiguration:
  config=kapp->config();

  // initialisiere Statusbar, Toolbar, ... und lese die Konfig.-Datei:
  initStatusBar();
  initDocument();
  initView();
  initAccel();
  initActions();
  readOptions();

  // setze die (gespeicherte) Hintergrundfarbe
  view->slotBackgroundColorChanged(properties->backgroundColor());

  QString playfile = properties->base() + properties->playFilename();

  // Soll Startup-Logo gezeigt werden ?
  if ( properties->startupLogo() && // nur wenn Logo angefordert wird, und
       !kapp->isRestored()          // nur wenn App. nicht wiederherstellt
                                    // wurde (Session Management)
     )
  {
    logo = new StartupLogo(0L, "logo");
    showLogo = (logo != 0L);
    if ( showLogo )
      logo->show();

  } // END if

  // erzeuge String-Liste mit den Aufloesungs-String fuer die Combo-Box:
  resList = new QStrList();

  // enable/disable Actions
  updateActions();

  // versuche herauszufinden, ob der Bildschirmschoner eingeschaltet ist:
  hasScreensaver = ScreensaverControl::getSaverStatus();

  // erzeuge Timer fuer das Wechseln der Dias:
  timerSlide = new QTimer(this);
  connect(timerSlide, SIGNAL(timeout()), this, SLOT(slotChangeSlide()));

  // lade Praesentation am Programmstart ?
  if ( properties->autoLoad() || playfilename != "" )
  {
    setCursor(KCursor::waitCursor());

    if ( playfilename == "" )
      openDocumentFile(playfile);
    else
      openDocumentFile(playfilename);

    setCursor(KCursor::arrowCursor());
  } // END if

  // entferne Startup-Logo (wenn angeszeigt)
  if ( showLogo )
  {
    sleep(2);  // 2 sec.

    logo->hide();
  } // END if showLogo

  // starte ggf. die geladene Praesentation
  if ( properties->autoLoad() && properties->autoStart() )
    slotControlPlay();

} // END KTourPresenterApp()


KTourPresenterApp::~KTourPresenterApp()
{
  windowList->remove(this);

} // END ~KTourPresenterApp()


void KTourPresenterApp::initAccel()
{
  // konfigurierbare Shortcuts:
  m_accel = new KAccel(this);

  m_accel->insertItem(i18n("More Brightness"),   ACTIONMOREBRIGHT, "B");
  m_accel->insertItem(i18n("Less Brightness"),   ACTIONLESSBRIGHT, "SHIFT+B");
  m_accel->insertItem(i18n("More Contrast"),     ACTIONMORECONTR,  "C");
  m_accel->insertItem(i18n("Less Contrast"),     ACTIONLESSCONTR,  "SHIFT+C");
  m_accel->insertItem(i18n("More Gamma"),        ACTIONMOREGAMMA,  "G");
  m_accel->insertItem(i18n("Less Gamma"),        ACTIONLESSGAMMA,  "SHIFT+G");
  m_accel->insertItem(i18n("Reset Modifier"),    ACTIONRESETMOD,   "R" );
  m_accel->insertItem(i18n("Play Presentation"), ACTIONPLAYPRES,   "P");
  m_accel->insertItem(i18n("Halt Presentation"), ACTIONHALTPRES,   "H");
  m_accel->insertItem(i18n("Stop Presentation"), ACTIONSTOPPRES,   "S");
  m_accel->insertItem(i18n("Start of this day or previous day"),
                                                 ACTIONPREVDAY,    "Down");
  m_accel->insertItem(i18n("Next day"),          ACTIONNEXTDAY,    "Up");
  m_accel->insertItem(i18n("Previous Slide"),    ACTIONPREVSLIDE,  "Left");
  m_accel->insertItem(i18n("Next Slide"),        ACTIONNEXTSLIDE,  "Right");
  m_accel->insertItem(i18n("Show Tour Report"),  ACTIONSHOWREPORT, "F2");
  m_accel->insertItem(i18n("Toggle Fullscreen mode"), ACTIONFULLSCREEN,
                                                            "CTRL+SHIFT+F");

  m_accel->connectItem(ACTIONMOREBRIGHT, view, SLOT(slotMoreBrightness()));
  m_accel->connectItem(ACTIONLESSBRIGHT, view, SLOT(slotLessBrightness()));
  m_accel->connectItem(ACTIONMORECONTR,  view, SLOT(slotMoreContrast()));
  m_accel->connectItem(ACTIONLESSCONTR,  view, SLOT(slotLessContrast()));
  m_accel->connectItem(ACTIONMOREGAMMA,  view, SLOT(slotMoreGamma()));
  m_accel->connectItem(ACTIONLESSGAMMA,  view, SLOT(slotLessGamma()));
  m_accel->connectItem(ACTIONRESETMOD,   view, SLOT(slotResetModifier()));
  m_accel->connectItem(ACTIONPLAYPRES,   this, SLOT(slotControlPlay()));
  m_accel->connectItem(ACTIONHALTPRES,   this, SLOT(slotControlPause()));
  m_accel->connectItem(ACTIONSTOPPRES,   this, SLOT(slotControlStop()));
  m_accel->connectItem(ACTIONPREVDAY,    this, SLOT(slotPrevDay()));
  m_accel->connectItem(ACTIONNEXTDAY,    this, SLOT(slotNextDay()));
  m_accel->connectItem(ACTIONPREVSLIDE,  this, SLOT(slotPrevSlide()));
  m_accel->connectItem(ACTIONNEXTSLIDE,  this, SLOT(slotNextSlide()));
  m_accel->connectItem(ACTIONSHOWREPORT, this, SLOT(slotControlReport()));
  m_accel->connectItem(ACTIONFULLSCREEN, this, SLOT(slotToggleFullscreen()));

  m_accel->readSettings();

} // END initAccel()


void KTourPresenterApp::initActions()
{
  fileOpen =
    KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());

  fileClose =
    KStdAction::close(this, SLOT(slotFileClose()), actionCollection());

  fileQuit = KStdAction::quit(this, SLOT(close()), actionCollection());
  
  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()),
                                        actionCollection());

  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()),
                                            actionCollection());
  
  controlStop = new KAction(i18n("S&top"), "player_stop",
                       m_accel->currentKey(ACTIONSTOPPRES),
                       this, SLOT(slotControlStop()), actionCollection(),
                       "control_stop");

  controlPlay = new KAction(i18n("&Start"), "1rightarrow",
                       m_accel->currentKey(ACTIONPLAYPRES),
                       this, SLOT(slotControlPlay()), actionCollection(),
                       "control_play");

  controlPause = new KAction(i18n("&Pause"), "player_pause",
                       m_accel->currentKey(ACTIONHALTPRES),
                       this, SLOT(slotControlPause()), actionCollection(),
                       "control_pause");

  controlPrevDay = new KAction(i18n("Previous Day"), "2leftarrow",
                       m_accel->currentKey(ACTIONPREVDAY),
                       this, SLOT(slotPrevDay()),
                       actionCollection(), "control_prevday");

  controlNextDay = new KAction(i18n("Next Day"), "2rightarrow",
                       m_accel->currentKey(ACTIONNEXTDAY),
                       this, SLOT(slotNextDay()), actionCollection(),
                       "control_nextday");

  controlPrevSlide = new KAction(i18n("Previous Slide"), "back",
                       m_accel->currentKey(ACTIONPREVSLIDE),
                       this, SLOT(slotPrevSlide()), actionCollection(),
                       "control_prevslide");

  controlNextSlide = new KAction(i18n("Next Slide"), "forward",
                       m_accel->currentKey(ACTIONNEXTSLIDE),
                       this, SLOT(slotNextSlide()), actionCollection(),
                       "control_nextslide");

  controlFullscreen = new KAction(i18n("Enable &fullscreen"),
                       "window_fullscreen",
                       m_accel->currentKey(ACTIONFULLSCREEN),
                       this, SLOT(slotToggleFullscreen()), actionCollection(),
                       "control_fullscreen");

  controlReport = new KAction(i18n("Show tour &report"), "package_editors",
                       m_accel->currentKey(ACTIONSHOWREPORT),
                       this, SLOT(slotControlReport()), actionCollection(),
                       "control_report");

  modMoreBrightness = new KAction(i18n("More Brightness"), "brightness+",
                       m_accel->currentKey(ACTIONMOREBRIGHT),
                       view, SLOT(slotMoreBrightness()), actionCollection(),
                       "mod_morebrightness");

  modLessBrightness = new KAction(i18n("Less Brightness"), "brightness-",
                       m_accel->currentKey(ACTIONLESSBRIGHT),
                       view, SLOT(slotLessBrightness()), actionCollection(),
                       "mod_lessbrightness");

  modMoreContrast = new KAction(i18n("More Contrast"), "contrast+",
                       m_accel->currentKey(ACTIONMORECONTR),
                       view, SLOT(slotMoreContrast()), actionCollection(),
                       "mod_morecontrast");

  modLessContrast = new KAction(i18n("Less Contrast"), "contrast-",
                       m_accel->currentKey(ACTIONLESSCONTR),
                       view, SLOT(slotLessContrast()), actionCollection(),
                       "mod_lesscontrast");

  modMoreGamma = new KAction(i18n("More Gamma"), "gamma+",
                       m_accel->currentKey(ACTIONMOREGAMMA),
                       view, SLOT(slotMoreGamma()), actionCollection(),
                       "mod_moregamma");

  modLessGamma = new KAction(i18n("Less Gamma"), "gamma-",
                       m_accel->currentKey(ACTIONLESSGAMMA),
                       view, SLOT(slotLessGamma()), actionCollection(),
                       "mod_lessgamma");

  modReset = new KAction(i18n("Reset Modifier"), "undo",
                       m_accel->currentKey(ACTIONRESETMOD),
                       view, SLOT(slotResetModifier()), actionCollection(),
                       "mod_reset");

  KStdAction::keyBindings(this, SLOT(slotConfigKeys()), actionCollection());

  KStdAction::configureToolbars(this, SLOT(slotConfigToolbar()),
                                actionCollection());

  KStdAction::preferences(this, SLOT(slotConfigure()), actionCollection());


  fileOpen->setStatusText(i18n("Opens an existing tour presentation"));
  fileClose->setStatusText(i18n("Closes the current tour presentation"));
  fileQuit->setStatusText(i18n("Quits KTourPresenter"));
  
  viewToolBar->setStatusText(i18n("Enables/disables the toolbar"));
  viewStatusBar->setStatusText(i18n("Enables/disables the statusbar"));
  
  controlPlay->setStatusText(i18n(ACTIONPLAYPRES));
  controlStop->setStatusText(i18n(ACTIONSTOPPRES));
  controlPause->setStatusText(i18n(ACTIONHALTPRES));
  controlReport->setStatusText(i18n(ACTIONSHOWREPORT));
  controlFullscreen->setStatusText(i18n(ACTIONFULLSCREEN));

  // Kontext-Menue:
  cmenu = new KPopupMenu(this, "contextMenu");

  ID_CONTEXT_PLAY =
    cmenu->insertItem(SmallIconSet("1rightarrow"), i18n("Start presentation"),
                      this, SLOT(slotControlPlay()),
                      m_accel->currentKey(ACTIONPLAYPRES));

  ID_CONTEXT_PAUSE =
    cmenu->insertItem(SmallIconSet("player_pause"), i18n("Pause presentation"),
                      this, SLOT(slotControlPause()),
                      m_accel->currentKey(ACTIONHALTPRES));

  ID_CONTEXT_STOP =
    cmenu->insertItem(SmallIconSet("player_stop"), i18n("Stop presentation"),
                      this, SLOT(slotControlStop()),
                      m_accel->currentKey(ACTIONSTOPPRES));

  ID_CONTEXT_SEP1 = cmenu->insertSeparator();

  ID_CONTEXT_NEXTDAY =
    cmenu->insertItem(SmallIconSet("2rightarrow"), i18n("Next day"),
                      this, SLOT(slotNextDay()),
                      m_accel->currentKey(ACTIONNEXTDAY));

  ID_CONTEXT_NEXTSLIDE =
    cmenu->insertItem(SmallIconSet("forward"), i18n("Next slide"),
                      this, SLOT(slotNextSlide()),
                      m_accel->currentKey(ACTIONNEXTSLIDE));

  ID_CONTEXT_PREVSLIDE =
    cmenu->insertItem(SmallIconSet("back"), i18n("Previous slide"),
                      this, SLOT(slotPrevSlide()),
                      m_accel->currentKey(ACTIONPREVSLIDE));

  ID_CONTEXT_PREVDAY =
    cmenu->insertItem(SmallIconSet("2leftarrow"), i18n("Previous day"),
                      this, SLOT(slotPrevDay()),
                      m_accel->currentKey(ACTIONPREVDAY));

  ID_CONTEXT_SEP2 = cmenu->insertSeparator();

  ID_CONTEXT_MOREBRIGHTNESS =
    cmenu->insertItem(SmallIconSet("brightness+"), i18n("More Brightness"),
                      view, SLOT(slotMoreBrightness()),
                      m_accel->currentKey(ACTIONMOREBRIGHT));

  ID_CONTEXT_LESSBRIGHTNESS =
    cmenu->insertItem(SmallIconSet("brightness-"), i18n("Less Brightness"),
                      view, SLOT(slotLessBrightness()),
                      m_accel->currentKey(ACTIONLESSBRIGHT));

  ID_CONTEXT_MORECONTRAST =
    cmenu->insertItem(SmallIconSet("contrast+"), i18n("More Contrast"),
                      view, SLOT(slotMoreContrast()),
                      m_accel->currentKey(ACTIONMORECONTR));

  ID_CONTEXT_LESSCONTRAST =
    cmenu->insertItem(SmallIconSet("contrast-"), i18n("Less Contrast"),
                      view, SLOT(slotLessContrast()),
                      m_accel->currentKey(ACTIONLESSCONTR));

  ID_CONTEXT_MOREGAMMA =
    cmenu->insertItem(SmallIconSet("gamma+"), i18n("More Gamma"),
                      view, SLOT(slotMoreGamma()),
                      m_accel->currentKey(ACTIONMOREGAMMA));

  ID_CONTEXT_LESSGAMMA =
    cmenu->insertItem(SmallIconSet("gamma-"), i18n("Less Gamma"),
                      view, SLOT(slotLessGamma()),
                      m_accel->currentKey(ACTIONLESSGAMMA));

  ID_CONTEXT_RESETMOD =
    cmenu->insertItem(SmallIconSet("undo"), i18n("Reset Modifier"),
                      view, SLOT(slotResetModifier()),
                      m_accel->currentKey(ACTIONRESETMOD));

  ID_CONTEXT_SEP3 = cmenu->insertSeparator();

  ID_CONTEXT_REPORT =
    cmenu->insertItem(SmallIconSet("package_editors"),
                      i18n("Show tour report"),
                      this, SLOT(slotControlReport()),
                      m_accel->currentKey(ACTIONSHOWREPORT));

  ID_CONTEXT_FULLSCR =
    cmenu->insertItem(SmallIconSet("window_fullscreen"),
                      i18n("Enable &fullscreen"),
                      this, SLOT(slotToggleFullscreen()),
                      m_accel->currentKey(ACTIONFULLSCREEN));

  ID_CONTEXT_SEP4 = cmenu->insertSeparator();

  ID_CONTEXT_QUIT =
    cmenu->insertItem(SmallIconSet("exit"), i18n("E&xit"),
                      this, SLOT(close()),
                      KStdAccel::quit().keyCodeQt());

  createGUI();

} // END initActions()


void KTourPresenterApp::initStatusBar()
{
  // Meldungen:
  m_statusLabel = new KStatusBarLabel("", ID_STATUS_MSG, statusBar(), "msg");
  m_statusLabel->setFixedHeight(m_statusLabel->sizeHint().height());
  m_statusLabel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
  m_statusLabel->setMargin(1);
  m_statusLabel->setLineWidth(0);
  m_statusLabel->setAlignment(AlignLeft | AlignVCenter);

  // Status-LED:
  stateLed = new KLed(QColor("red"), KLed::Off, KLed::Sunken, KLed::Circular,
                      statusBar(), "led");
  stateLed->setFixedHeight(m_statusLabel->height());
  stateLed->setFixedWidth(m_statusLabel->height());

  // Aufloesungs-Combobox:
  resCombo = new QComboBox(false, statusBar(), "rescombo");
  resCombo->setMinimumWidth(170); // *** FIXME: feste Groesse, ugly !!!
  resCombo->setFocusPolicy(NoFocus);
  connect(resCombo, SIGNAL(activated(const QString&)),
          this,     SLOT(slotSelectRes(const QString&)));

  statusBar()->addWidget(m_statusLabel, 1, false);
  statusBar()->addWidget(stateLed,      0, true);
  statusBar()->addWidget(resCombo,      0, true);

  QString stateWT = "Status of the resentation.\n"
                    "Stopped presentation (red), "
                    "running presentation (green) and "
                    "paused presentation (yellow).";
  QWhatsThis::add(stateLed, i18n(stateWT.data()));

  /*
     *** FIXME: erstmal auskommentiert, da in der Tour CD 2002 (Vom Bodensee
                in die Dolomiten) nicht die Aufloesung, sondern der Teil der
                Dia-Show (1 bis 3) gewaehlt wird.

  QToolTip::add(resCombo, i18n("Resolution"));
  QWhatsThis::add(resCombo,
    i18n("Select a resolution for the images of the presentation."));
  */
  
} // END initStatusBar()


void KTourPresenterApp::initDocument()
{
  doc = new KTourPresenterDoc(this);
  doc->newDocument();
} // END initDocument()


void KTourPresenterApp::initView()
{ 
  view = new KTourPresenterView(this);

  // Signal durch Maus-Rad:
  connect(view, SIGNAL(sigNext()), this, SLOT(slotNextSlide()));
  connect(view, SIGNAL(sigPrev()), this, SLOT(slotPrevSlide()));

  setCentralWidget(view);	
  doc->addView(view);

  // Praesentationsnamen als Fenstertitel:
  setCaption(doc->presentationName(), false);

  // wenn der Slide bewegt wird
  connect(view, SIGNAL(sigNewSliderPos(int)), this, SLOT(slotSeek(int)));

  //
  connect(this, SIGNAL(sigClose()), view, SLOT(slotReset()));

} // END initView()


void KTourPresenterApp::openDocumentFile(const KURL& url)
{
  QString urlpath = url.path();
  QString message = i18n("Opening file") + " " + urlpath;
  slotStatusMsg(message);

#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourPresenterApp::openDocumentFile(" << urlpath << ")" << endl;
#endif

  // separiere aus vollstaendigen Dateinamen den Pfad fuer
  // Properties::base(), und setze diesen:
  int idx = urlpath.findRev('/');
  if ( idx == -1 )
  {
    properties->base("/");
  }
  else
  {
    properties->base(urlpath.left(idx));
    properties->playFilename(urlpath.right(urlpath.length() - idx - 1));
  } // END if

  try
  {
    if ( !url.isEmpty() && doc->openDocument(KURL(urlpath)) )
    {
      setCaption(doc->presentationName(), false);

      // Zeige die Default-Grafik der Praesentation, wenn existient:
      QString startIMG = properties->base() +
                         properties->defaultImageFilename();
      if ( QFile::exists(startIMG) )
      {
        view->loadImage(startIMG.data());
        view->showImage();
      } // END if QFile::exist()

      // Praesentation geladen, aber nicht gestartet:
      status = stopped;

      // enable/disable Actions:
      updateActions();
    
      // Setze die Aufloesungs-Combo:
      doc->resStrList( *resList );
      resCombo->insertStrList(resList);

      // Zeige die Gesamtzeit der Praesentation:
      view->updateTotalTime(doc->totalTime());

      // Setzte den Maximalwert fuer den Schieber:
      view->setSliderMax(doc->totalTime()-1);
    
      // Eine Datei wurde geoeffnet, schalte den Fortschrittsschieber ein:
      view->setEnabled(true);

      // Erzeuge ggf. den Sound-Player:
      if ( !player )
        player = new SoundPlayer();

      // und oeffne die Sound-Datei:
      player->openFile(doc->soundFilename());
      player->play(); player->pause(); player->seek(1);

    } // END if !url.isEmpty() && doc->openDocument()

  } // END try
  catch ( Schedule::ContentError& err)
  {
    KMessageBox::sorry(0, err.what(),
                       i18n("Error while reading schedule file"));

  } // END ContentError
  catch ( Schedule::CanNotOpenFile& err)
  {
    KMessageBox::sorry(0, err.what(),
                       i18n("Error while reading schedule file"));
  } // END CanNotOpenFile
  catch ( ... )
  {
    KMessageBox::sorry(0,
                       i18n("Error while reading schedule file"),
                       i18n("Error while reading schedule file"));
  } // END ...

  slotStatusMsg(i18n("Ready."));
} // END openDocumentFile()


KTourPresenterDoc *KTourPresenterApp::getDocument() const
{
  return doc;
} // getDocument()


void KTourPresenterApp::saveOptions()
{
  // [General Options]
  config->setGroup("General Options");
  config->writeEntry("Geometry",            size());
  config->writeEntry("Show Main Toolbar",   viewToolBar->isChecked());
  config->writeEntry("Show Statusbar",      viewStatusBar->isChecked());
  config->writeEntry("Main Toolbar Position",
                                     (int)toolBar("mainToolBar")->barPos());
  config->writeEntry("Main Toolbar Icon Text",
                                     (int)toolBar("mainToolBar")->iconText());
  config->writeEntry("Show Logo",           properties->startupLogo());
  config->writeEntry("Disable Screensaver", properties->disableScreensaver());

  // [Slideshow Parameters]
  config->setGroup("Slideshow Parameters");
  config->writeEntry("Auto Load",             properties->autoLoad());
  config->writeEntry("Auto Start",            properties->autoStart());
  config->writeEntry("Auto Quit",             properties->autoQuit());
  config->writeEntry("Background Color",      properties->backgroundColor());
  config->writeEntry("Fullscreen",            properties->fullscreen());
  config->writeEntry("Hide Mouse",            properties->hideMouse());
  config->writeEntry("Hide Delay",            properties->hideDelay());
  config->writeEntry("Extended Slide Change", properties->extSlideChange());

  // [File and Directory Names]
  config->setGroup("File and Directory Names");
  config->writeEntry("Base Path",        properties->base());
  config->writeEntry("Play File",        properties->playFilename());
  config->writeEntry("Default Image",    properties->defaultImageFilename());
  config->writeEntry("Report",           properties->reportFilename());

  // [Imlib Parameters]
  config->setGroup("Imlib Parameters");
  config->writeEntry("Own Palette",      properties->ownPalette());
  config->writeEntry("Fast Remap",       properties->fastRemap());
  config->writeEntry("Fast Render",      properties->fastRender());
  config->writeEntry("Dither 16bit",     properties->dither16bit());
  config->writeEntry("Dither 8bit",      properties->dither8bit());
  config->writeEntry("Gamma Value",      properties->gamma());
  config->writeEntry("Brightness Value", properties->brightness());
  config->writeEntry("Contrast Value",   properties->contrast());
  config->writeEntry("Gamma Factor",     properties->gammaFactor());
  config->writeEntry("Brightness Factor",properties->brightnessFactor());
  config->writeEntry("Contrast Factor",  properties->contrastFactor());
  config->writeEntry("Max Cache",        properties->maxCache());
  config->writeEntry("Up Scale",         properties->upScale());
  config->writeEntry("Down Scale",       properties->downScale());
  config->writeEntry("Max Up Scale",     properties->maxUpScale());

  // [Keys]
  m_accel->writeSettings(kapp->config());

  config->sync();
} // END saveOpetions()


void KTourPresenterApp::readOptions()
{
  // [General Options]
  config->setGroup("General Options");

  // config version
  int configVersion = config->readNumEntry("Configuration File Version",
                                           CONFIGFILEVERSION);
  // Dummy-Vergleich; Wichtig vielleicht erst in spaeteren Versionen, wenn
  // sich das Format der Konfig.-Datei aendert; siehe ktourpresenter.h,
  // unten.
  if ( configVersion < 1 )
  { } else { } // END if

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Main Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();

  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos =
    (KToolBar::BarPosition) config->readNumEntry("Main Toolbar Position",
                                                 KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);

  // bar icon text
  KToolBar::IconText iconText;
  iconText =
    (KToolBar::IconText) config->readNumEntry("Main Toolbar Icon Text",
                                                 KToolBar::IconOnly);
  toolBar("mainToolBar")->setIconText(iconText);

  // Geometry
  QSize size=config->readSizeEntry("Geometry");
  if(!size.isEmpty())
    resize(size);

  // show startup logo:
  bool logoFlag = config->readBoolEntry("Show Logo", true);
  properties->startupLogo(logoFlag);

  // disable screensaver:
  bool ssaverFlag = config->readBoolEntry("Disable Screensaver", false);
  properties->disableScreensaver(ssaverFlag);

  // [Notification Messages]
  config->setGroup("Notification Messages");

  bool confirmExit = config->readBoolEntry("Confirm Exit", true);
  properties->confirmExit(confirmExit);

  bool screenSaverWarning =
    config->readBoolEntry("Confirm Disable Screensaver Warning", true);
  properties->screenSaverWarning(screenSaverWarning);


  // [Slideshow Parameters]
  config->setGroup("Slideshow Parameters");
  
  bool autoLoadFlag = config->readBoolEntry("Auto Load", false);
  properties->autoLoad(autoLoadFlag);
  
  bool autoStartFlag = config->readBoolEntry("Auto Start", false);
  properties->autoStart(autoStartFlag);

  bool autoQuitFlag = config->readBoolEntry("Auto Quit", false);
  properties->autoQuit(autoQuitFlag);

  QColor backgroundC = config->readColorEntry("Background Color",
                                               new QColor("black"));
  properties->backgroundColor(backgroundC);

  bool fullscreenFlag = config->readBoolEntry("Fullscreen", false);
  properties->fullscreen(fullscreenFlag);

  bool hideMouseFlag = config->readBoolEntry("Hide Mouse", false);
  properties->hideMouse(hideMouseFlag);

  int hideDelay = config->readNumEntry("Hide Delay", 3000);
  properties->hideDelay(hideDelay);

  bool extSlideChange = config->readBoolEntry("Extended Slide Change", false);
  properties->extSlideChange(extSlideChange);


  // [File and Directory Names]
  config->setGroup("File and Directory Names");

  QString base = config->readEntry("Base Path", QDir::homeDirPath());
  properties->base(base);
  
  QString playfile = config->readEntry("Play File", "play.ini");
  properties->playFilename(playfile);

  QString defaultImg = config->readEntry("Default Image", "default.jpg");
  properties->defaultImageFilename(defaultImg);
  
  QString report = config->readEntry("Report", "bericht.htm");
  properties->reportFilename(report);


  // [Imlib Parameters]
  config->setGroup("Imlib Parameters");

  bool ownPalette = config->readBoolEntry("Own Palette", true);
  properties->ownPalette(ownPalette);

  bool fastRemap = config->readBoolEntry("Fast Remap", true);
  properties->fastRemap(fastRemap);

  bool fastRender = config->readBoolEntry("Fast Render", true);
  properties->fastRender(fastRender);

  bool dither16 = config->readBoolEntry("Dither 16bit", false);
  properties->dither8bit(dither16);

  bool dither8 = config->readBoolEntry("Dither 8bit", true);
  properties->dither8bit(dither8);

  int gamma = config->readNumEntry("Gamma Value", 0);
  properties->gamma(gamma);

  int brightness = config->readNumEntry("Brightness Value", 0);
  properties->brightness(brightness);

  int contrast = config->readNumEntry("Contrast Value", 0);
  properties->contrast(contrast);

  unsigned int gammaFactor =
    config->readUnsignedNumEntry("Gamma Factor", 10);
  properties->gammaFactor(gammaFactor);

  unsigned int brightnessFactor =
    config->readUnsignedNumEntry("Brightness Factor", 10);
  properties->brightnessFactor(brightnessFactor);

  unsigned int contrastFactor =
    config->readUnsignedNumEntry("Contrast Factor", 10);
  properties->contrastFactor(contrastFactor);

  unsigned int maxCache = config->readUnsignedNumEntry("Max Cache", 32768);
  properties->maxCache(maxCache);

  bool upScale = config->readBoolEntry("Up Scale", false);
  properties->upScale(upScale);

  bool downScale = config->readBoolEntry("Down Scale", true);
  properties->downScale(downScale);

  int maxUpScale = config->readNumEntry("Max Up Scale", 3);
  properties->maxUpScale(maxUpScale);

  // [Keys]
  m_accel->readSettings(kapp->config());
} // END readOptions()


void KTourPresenterApp::saveProperties(KConfig *_cfg)
{
  KURL url=doc->URL();	
  _cfg->writeEntry("filename", url.url());
  QString tempname = kapp->tempSaveName(url.url());
  QString tempurl= KURL::encode_string(tempname);
  //KURL _url(tempurl);
  
  /* *** FIXME: more saved properties !!!
                - if needed the mount device with the presentation on it
                  (e.g. /cdrom)
                - name of the primary config file (e.g. play.ini)
                - choosen resolution or the special config file (e.g.
                  mittel.ply)
                - position in the presentation
   */
} // END saveProperties()


void KTourPresenterApp::readProperties(KConfig* _cfg)
{
  QString filename = _cfg->readEntry("filename", "");
  KURL url(filename);
  
  if ( !filename.isEmpty() )
  {
    doc->openDocument(url);
    setCaption(doc->presentationName(), false);
  }
  
  // *** FIXME: see saveProperties() !!!
  
} // END readProperties()


bool KTourPresenterApp::queryClose()
{
  slotStatusMsg(i18n("Exiting..."));

  bool confirmFlag = properties->confirmExit();
  QString text = i18n("Do you really want to quit this application ?");
  QString caption = i18n("Quit");
  QString button  = i18n("Quit");

  QString dontAskAgainName = "Confirm Exit";
  if ( ( !confirmFlag ) ||
       ( confirmFlag && KMessageBox::warningContinueCancel(0, text,
                      caption, button, dontAskAgainName, true)
                    == KMessageBox::Continue )
     )
  {

    // wenn Bildschirmschoner (immernoch) ausgeschaltet ist, dann mache dies
    // rueckgaengig:
    if ( hasScreensaver )
      ScreensaverControl::setEnabled(true);

    return true;
  } // END if

  slotStatusMsg(i18n("Ready."));

  return false;

} // END queryClose()


void KTourPresenterApp::setComboBoxEnabled(bool flag)
{
  resCombo->setEnabled(flag);
} // END setComboBoxEnabled()


void KTourPresenterApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));

  QString dir = QString::null;
  
  if ( QDir(properties->base()).exists() )
  {
    dir = properties->base();
  }

  KURL url=KFileDialog::getOpenURL(dir,
        i18n("play.ini|Play File (play.ini)\n*|All Files"), this,
        i18n("Open File..."));

  try
  {
    if ( !url.isEmpty() )
    {
      if ( status != none )
      {
        player->closeFile();
        doc->deleteContents();
      } // END if status

      resCombo->clear();
      openDocumentFile(url);
      view->updateActTime(0);
    } // END if url
  }
  catch (Schedule::ContentError())
  {
    KMessageBox::sorry(0L,
                       i18n("I can not read the file %1 ! Maybe\n"
                            "this is not a valid schedule file.\n"
                            "Please choose another file.").arg(url.path()),
                       i18n("Error in %1").arg(url.path()));
    status = none;

    return;
  }
  catch (Schedule::CanNotOpenFile(const QString&))
  {
    KMessageBox::sorry(0L,
                       i18n("I can not open the file %1 ! Maybe\n"
                            "this file doesn't exist.\n"
                            "Please choose another file.").arg(url.path()),
                       i18n("Can not open %1").arg(url.path()));
    status = none;

    return;
  }
  catch (...)
  {
    // *** FIXME: Fehlermeldung !

    status = none;

    return;
  } // END try-catch

  status = stopped;

  // starte ggf. die geladene Praesentation:
  if ( properties->autoStart() )
    slotControlPlay();
  else
    slotStatusMsg(i18n("Ready."));

  setComboBoxEnabled(true);

} // END slotFileOpen()


void KTourPresenterApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));

  if ( status == running || status == paused )
    slotControlStop();

  // schliesse das Dokument
  doc->closeDocument();
  resCombo->clear();

  view->updateActTime(0);
  view->setSliderMax(0);

  player->closeFile();
  emit sigClose();
  status = none;

  setCaption("KTourPresenter");

  updateActions();
  
  setComboBoxEnabled(false);

  slotStatusMsg(i18n("Ready."));

} // END slotFileClose()


void KTourPresenterApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggling toolbar..."));
  
  if ( !viewToolBar->isChecked() )
    toolBar("mainToolBar")->hide();
  else
    toolBar("mainToolBar")->show();

  slotStatusMsg(i18n("Ready."));

} // END slotViewToolBar()


void KTourPresenterApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  
  if ( !viewStatusBar->isChecked() )
    statusBar()->hide();
  else
    statusBar()->show();

  slotStatusMsg(i18n("Ready."));

} // END slotViewStatusBar()


void KTourPresenterApp::slotStatusMsg(const QString& text)
{
  statusBar()->clear();
  m_statusLabel->setText(text);
} // END slotStatusMsg()


void KTourPresenterApp::slotControlPlay()
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourPresenterApp::slotControlPlay()" << endl;
#endif

  switch ( status )
  {
    case stopped:
    {
      // Praesentation beginnt von vorne
      // -------------------------------

      // soll im Vollbildmodus gestartet werden ?
      if ( properties->fullscreen() &&
           view->viewMode() == KTourPresenterView::Window )
      {
        this->slotToggleFullscreen();
      } // END if

      // soll versucht werden, den Bildschirmschoner auszuschalten ?
      if ( hasScreensaver && properties->disableScreensaver() )
        ScreensaverControl::setEnabled(false);
    
      // soll der Mauszeiger nach bestimmter Zeit 'verschwinden' ?
      view->setAutoHideEnabled(properties->hideMouse());

      // setze Ablaufplan zurueck, Ablaufzeiger verweist auf Default-Grafik
      doc->reset();

      // loesche einen ggf. vorher erzeugten Schnappschuss
      snapshot.erase();

      // eine Aenderung der Ausfloesung ist beim Abspielen der Praesentation
      // nicht moeglich !
      setComboBoxEnabled(false);

      // starte Sound-Ausgabe:
      slotPlaySound();

      // Wann soll das erste Dia angezeigt werden ?
      Slide& s = doc->slide(1);
      int changeTime;
      if ( properties->extSlideChange() )
        changeTime = s.absTime() - (s.duration()/2);
      else
        changeTime = s.absTime() + s.slideChangeOffset();

      timerSlide->start(changeTime, true);
#ifndef NDEBUG
      cerr << "[" << Globals::currTime() << "]: "
           << "KTourPresenterApp::slotControlPlay(); 1. Dia ab "
           << changeTime << " msec." << endl;
#endif
      
      // cache das erste Dia der Praesentation, aktuell bleibt aber bisher
      // slide[0], die Default-Grafik
      cacheNextImage();

      // nun 'laeuft' die Praesentation
      status = running;

      break;
    } // case running

  case paused:
    {
      // eine Praesentation wurde vorher angehalten
      // ------------------------------------------

      // Stelle den Schnappschuss wieder her
      if ( snapshot.isValid() )
      {
        // Spule die Sound-Datei zur Schnappschusszeit
        slotSeek(snapshot.absTime());

        // fuehre Sound-Ausgabe weiter
        if ( player->isPlaying() || player->isPaused() )
        {
          player->pause();
          player->seek(snapshot.absTime());
        }
        else
        {
          // bisher wurde Sound-Ausgabe noch nicht gestartet, Praesentation
          // kam durch Verschieben des Sliders vor dem Start in den Pause-
          // Modus
          player->play();
          //player->seek(snapshot.absTime());
          player->seek(doc->slide().absTime());
        } // END if

        // starte Timer fuer die Restzeit des Dias
        timerSlide->start(snapshot.remainingTime(), true);

        // loesche den vorher erzeugten Schnappschuss
        snapshot.erase();

        // nun 'laeuft' die Praesentation wieder
        status = running;

        // soll der Mauszeiger nach bestimmter Zeit 'verschwinden' ?
        view->setAutoHideEnabled(properties->hideMouse());

      } // END if

      break;
    } // case paused:

  default:
    break;
  } // END switch

  updateActions();

  slotStatusMsg(i18n("Playing presentation..."));

} // END slotControlPlay()


void KTourPresenterApp::slotControlPause()
{
  switch ( status )
  {
    case running:
      {
        // Setzte Schnappschuss:
        if ( player->isPlaying() )
          snapshot.set(doc, doc->slideIdx(), player->getPos());
        else
          snapshot.set(doc, 0, 0);

        // halte Sound-Player an
        player->pause();

        // Stoppe Timer fuer Diawechsel
        timerSlide->stop();

        status = paused;

        slotStatusMsg(i18n("Delay presentation."));

        // einen 'verschwundenen' Mauszeiger wieder zeigen
        if ( properties->hideMouse() )
          view->setAutoHideEnabled(false);

        break;
      } // case running

    case paused:
      {
        slotControlPlay();

        break;
      } // case paused

    default:
    {
      // Kann das vorkommen ?

      break;
    } // default

  } // END switch

  updateActions();

} // END slotControlPause()


void KTourPresenterApp::slotControlStop()
{
  // beende Timer fuer Dia-Wechsel:
  timerSlide->stop();

  // beende Abspielen der Sound-Daten:
  player->stop();

  // wechsle ggf. vom Fullscreen in den Window Modus
  if ( view->viewMode() == KTourPresenterView::Fullscreen )
    this->slotToggleFullscreen();

  // soll versucht werden, den Bildschirmschoner wieder einzuschalten ?
  if ( hasScreensaver )
    ScreensaverControl::setEnabled(true);

  // einen 'verschwundenen' Mauszeiger wieder zeigen
  if ( properties->hideMouse() )
    view->setAutoHideEnabled(false);

  // Zeige Default-Grafik dieser Praesentation:
  QString defImg = properties->base() + properties->defaultImageFilename();
  view->loadImage(defImg.data());

  // Setze Fortschrittsanzeige zurueck:
  view->updateActTime(0);
  view->updateSlider(0);

  // Eine Wahl der Aufloesung nun wieder moeglich:
  setComboBoxEnabled(true);

  // Setze Dokument zurueck:
  doc->reset();

  view->cacheImage(doc->schedule().slide(1).picFilename());

  // Loesche den Schnappschuss:
  snapshot.erase();

  status = stopped;

  updateActions();

  slotStatusMsg(i18n("Ready."));

} // END slotControlStop()


void KTourPresenterApp::slotControlReport()
{
  QString url = properties->base()+properties->reportFilename();
    
  KRun::run(properties->browserName(), QStringList(url));
} // END slotControlReport()


void KTourPresenterApp::slotConfigKeys()
{
  KKeyDialog::configureKeys(m_accel);
} // END slotConfigKeys()


void KTourPresenterApp::slotConfigToolbar()
{
/*
  KEditToolbar dlg(factory());

  if (dlg.exec())
    createGUI();
*/

  KEditToolbar dlg( actionCollection() );
  connect(&dlg, SIGNAL(newToolbarConfig()),
          this, SLOT(slotNewToolbarConfig()));

  if ( dlg.exec() )
    createGUI();

} // END slotConfigToolbar()


void KTourPresenterApp::slotConfigure()
{
  if ( configdialog == 0L )
  {
    configdialog = new ConfigureDialog(topLevelWidget(), "configdialog",
                                       m_accel, this, false);
    connect(configdialog, SIGNAL(sigBackgroundColorChanged(const QColor&)),
            view,         SLOT(slotBackgroundColorChanged(const QColor&)));
    connect(configdialog, SIGNAL(sigModifierChanged()),
            view,         SLOT(slotModifierChanged()));
    connect(configdialog, SIGNAL(sigToolbarChanged()),
                          SLOT(slotToolbarChanged()));
    connect(configdialog, SIGNAL(sigDisableScreensaver(bool)),
                          SLOT(slotDisableScreensaver(bool)));
    connect(configdialog, SIGNAL(sigHideMouseChanged(bool)),
            view,         SLOT(slotHideMouseChanged(bool)));
  }
  else
  {
    // Properties haben sich vielleicht geaendert, lasse Eingabe-Elemente
    // mit den aktuellen Werten aus Properties updaten
    configdialog->setValues();
  } // END if

  configdialog->show();

} // END slotConfigure()


void KTourPresenterApp::slotSelectRes(const QString& newres)
{
  // wechsle zur gewaehlten Aufloesung / zum Show-Teil (wenn vorhanden)
  if ( doc->setScheduleByName(newres) )
  {
#ifndef NDEBUG
    cerr << "KTourPresenterApp::slotSelectRes(" << newres << "); "
         << "Soundfile=" << doc->soundFilename() << endl;
#endif

    // *** FIXME: durch die Tour-CD 2002 ist ein Wechsel der Aufloesung / des
    //            Teils nur beim Stopped-Status moeglich
    // lade die ggf. neue Sound-Datei:
    player->openFile(doc->soundFilename());
    player->seek(1);

    // passe Gesamtsabpielzeit und Slider-Groesse an:
    view->updateTotalTime(doc->totalTime());
    view->setSliderMax(doc->totalTime()-1);
    view->updateActTime(0);

    QString imageName;
    if ( status == stopped )
      imageName = properties->base() + properties->defaultImageFilename();
    else
      imageName = doc->slide().picFilename();

    // lade die Grafik der neu gewaehlten Aufloesung
    if ( !properties->hideMouse() )
      setCursor(KCursor::waitCursor());
    view->loadImage(imageName);
    if ( !properties->hideMouse() )
      setCursor(KCursor::arrowCursor());
  }
  else
  {
    // es gibt keine passende Aufloesung / keinen passenden Show-Teil !

    // *** FIXME: was zu tun ???

  } // END if doc->setScheduleByName(newres)
} // END slotSelectRes()


void KTourPresenterApp::slotPlaySound()
{
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourPresenterApp::slotPlaySound()" << endl;
#endif

  player->play();

} // END slotPlaySound()


void KTourPresenterApp::slotSeek(int msec)
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::waitCursor());

  int seekTime = doc->seek(msec);
  
  // aendere actTimeLabel:
  view->updateActTime(doc->slide().absTime());

  // lade neue Grafik:
  view->loadImage(doc->slide().picFilename());
  view->showImage();
  
  player->seek(seekTime);

  if ( status == running )
  {
    timerSlide->start(doc->slide().duration(), true);
    int slideIdx = doc->schedule().currSlideIdx();
    view->cacheImage(doc->schedule().slide(slideIdx+1).picFilename());
  } // END if
  else
  {
    if ( (status == stopped) || (status == paused) )
    {
      // das Verschieben des Sliders bei einer nicht gestarteten (stopped)
      // Praesentation laesst den Status in paused wechseln => von der neuen
      // Position muss ein Snapshot gesetzt werden, analoges gilt fuer letztes
      // auch wenn der Zustand schon paused war

      if ( status == stopped )
      {
        // soll versucht werden, den Bildschirmschoner auszuschalten ?
        if ( hasScreensaver && properties->disableScreensaver() )
          ScreensaverControl::setEnabled(false);
      } // END if

      status = paused;

      snapshot.set(doc, doc->slideIdx(), doc->slide().absTime());
      player->seek(doc->slide().absTime());

      // speichere das naechste Dia ziwschen
      doc->next();
      view->cacheImage(doc->slide().picFilename());
      doc->prev();

      updateActions();
    } // END if
  } // END if

  if ( !properties->hideMouse() )
    setCursor(KCursor::arrowCursor());

} // END slotSeek()


void KTourPresenterApp::slotShowPopup(const QPoint& pos)
{
  cmenu->popup(pos);
} // END slotShowPopup()


void KTourPresenterApp::slotLastWindowClosed()
{
  saveOptions();
} // END slotLastWindowClosed()


void KTourPresenterApp::slotNewURL(const KURL& url)
{
#ifndef NDEBUG
  cerr << "KTourPresenterApp::slotNewURL(" << url.path() << ")" << endl;
#endif

  // laeuft noch eine Dia-Show (oder auch nur angehalten) ?
  if ( status == running || status == paused )
  {
    QString text = i18n("There is another slide show running ! To load the "
                        "new own you had to stop the old own.");
    QString caption = i18n("Another slide show is running");
    if ( KMessageBox::questionYesNo(0, text, caption) == KMessageBox::Yes )
    {
      // beende Timer fuer Dia-Wechsel:
      timerSlide->stop();

      // schliesse Dokument und Sound-Ausgabe
      doc->closeDocument();
      player->closeFile();
      emit sigClose();

      // soll versucht werden, den Bildschirmschoner wieder einzuschalten ?
      if ( hasScreensaver )
        ScreensaverControl::setEnabled(true);

      // einen 'verschwundenen' Mauszeiger wieder zeigen
      if ( properties->hideMouse() )
        view->setAutoHideEnabled(false);

      // Setze Fortschrittsanzeige zurueck:
      view->updateActTime(0);
      view->updateSlider(0);

      // Eine Wahl der Aufloesung nun wieder moeglich:
      setComboBoxEnabled(true);

      // Loesche den Schnappschuss:
      snapshot.erase();

      status = stopped;

      updateActions();

    } // END if KMessageBox

  } // END if status

  // nur wenn kleine Dia-Show laeuft kann die neue URL geladen werden
  if ( status == stopped || status == none )
  {
    openDocumentFile(url);
  } // END if

} // END slotNewURL()


void KTourPresenterApp::updateActions()
{
  // setze Actions abhaengig vom Status
  if ( status != none )
  {
    bool existReport =
      QFile::exists(properties->base() + properties->reportFilename());

    cmenu->setItemEnabled(ID_CONTEXT_REPORT,         existReport);
    cmenu->setItemEnabled(ID_CONTEXT_PREVDAY,        true);
    cmenu->setItemEnabled(ID_CONTEXT_NEXTDAY,        true);
    cmenu->setItemEnabled(ID_CONTEXT_PREVSLIDE,      true);
    cmenu->setItemEnabled(ID_CONTEXT_NEXTSLIDE,      true);
    cmenu->setItemEnabled(ID_CONTEXT_FULLSCR,        true);
    cmenu->setItemEnabled(ID_CONTEXT_MOREBRIGHTNESS, true);
    cmenu->setItemEnabled(ID_CONTEXT_LESSBRIGHTNESS, true);
    cmenu->setItemEnabled(ID_CONTEXT_MORECONTRAST,   true);
    cmenu->setItemEnabled(ID_CONTEXT_LESSCONTRAST,   true);
    cmenu->setItemEnabled(ID_CONTEXT_MOREGAMMA,      true);
    cmenu->setItemEnabled(ID_CONTEXT_LESSGAMMA,      true);
    cmenu->setItemEnabled(ID_CONTEXT_RESETMOD,       true);

    fileClose->setEnabled(true);
    controlReport->setEnabled(existReport);
    controlPrevDay->setEnabled(true);
    controlNextDay->setEnabled(true);
    controlPrevSlide->setEnabled(true);
    controlNextSlide->setEnabled(true);
    modMoreBrightness->setEnabled(true);
    modLessBrightness->setEnabled(true);
    modMoreContrast->setEnabled(true);
    modLessContrast->setEnabled(true);
    modMoreGamma->setEnabled(true);
    modLessGamma->setEnabled(true);
    modReset->setEnabled(true);
  } // END if

  switch ( status )
  {
    case running:
      {
        fileOpen->setEnabled(false);

        controlPlay->setEnabled(false);
        controlStop->setEnabled(true);
        controlPause->setEnabled(true);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  false);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  true);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, true);

        setComboBoxEnabled(false);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("green"));
        QToolTip::add(stateLed, i18n("Status: Running"));

        break;
      } // case running

    case paused:
      {
        fileOpen->setEnabled(false);

        controlPlay->setEnabled(true);
        controlStop->setEnabled(true);
        controlPause->setEnabled(true);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  true);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  true);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, true);

        // *** FIXME: Da neuere Praesentationen nicht unterschiedliche
        //            Aufloesungenen, sondern mehrere Teile beinhalten
        //            (die durch die Combo-Box gewechselt werden), wird
        //            die Benutzung der Combo-Box im Moment wieder nur im
        //            Stopped-Status erlaubt.
        //setComboBoxEnabled(true);
        setComboBoxEnabled(false);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("yellow"));
        QToolTip::add(stateLed, i18n("Status: Paused"));

        break;
      } // case paused:

    case stopped:
      {
        fileOpen->setEnabled(true);

        controlPlay->setEnabled(true);
        controlStop->setEnabled(false);
        controlPause->setEnabled(false);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  true);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  false);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, false);

        setComboBoxEnabled(true);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("red"));
        QToolTip::add(stateLed, i18n("Status: Stopped"));

        break;
      } // END stopped:

    case none:
      {
        fileOpen->setEnabled(true);
        fileClose->setEnabled(false);
        controlPlay->setEnabled(false);
        controlStop->setEnabled(false);
        controlPause->setEnabled(false);
        controlFullscreen->setEnabled(false);
        controlReport->setEnabled(false);
        controlPrevDay->setEnabled(false);
        controlNextDay->setEnabled(false);
        controlPrevSlide->setEnabled(false);
        controlNextSlide->setEnabled(false);
        modMoreBrightness->setEnabled(false);
        modLessBrightness->setEnabled(false);
        modMoreContrast->setEnabled(false);
        modLessContrast->setEnabled(false);
        modMoreGamma->setEnabled(false);
        modLessGamma->setEnabled(false);
        modReset->setEnabled(false);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,           false);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,           false);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE,          false);
        cmenu->setItemEnabled(ID_CONTEXT_REPORT,         false);
        cmenu->setItemEnabled(ID_CONTEXT_PREVDAY,        false);
        cmenu->setItemEnabled(ID_CONTEXT_NEXTDAY,        false);
        cmenu->setItemEnabled(ID_CONTEXT_PREVSLIDE,      false);
        cmenu->setItemEnabled(ID_CONTEXT_NEXTSLIDE,      false);
        cmenu->setItemEnabled(ID_CONTEXT_FULLSCR,        false);
        cmenu->setItemEnabled(ID_CONTEXT_MOREBRIGHTNESS, false);
        cmenu->setItemEnabled(ID_CONTEXT_LESSBRIGHTNESS, false);
        cmenu->setItemEnabled(ID_CONTEXT_MORECONTRAST,   false);
        cmenu->setItemEnabled(ID_CONTEXT_LESSCONTRAST,   false);
        cmenu->setItemEnabled(ID_CONTEXT_MOREGAMMA,      false);
        cmenu->setItemEnabled(ID_CONTEXT_LESSGAMMA,      false);
        cmenu->setItemEnabled(ID_CONTEXT_RESETMOD,       false);

        setComboBoxEnabled(false);

        stateLed->setState(KLed::Off);
        QToolTip::add(stateLed, i18n("Status: No presentation"));

        break;
      } // case none:

    default:
      { break; }

  } // END switch

} // END updateActions()


void KTourPresenterApp::slotChangeSlide()
{
  // erstmal die Verschiebung des vorherigen Dias sichern
  int lastOffset = doc->slide().slideChangeOffset();
  
  // nun soll das naechste Dia angezeigt werden
  doc->next();

  // (nun) aktuelles Dia:
  Slide& currSlide = doc->slide();
  
#ifndef NDEBUG
  cerr << "[" << Globals::currTime() << "]: "
       << "KTourPresenterApp::slotChangeSlide(" << currSlide.picFilename()
       << ")" << endl;
#endif

  // die Sound-Ausgabe beginnt erst mit der Darstellung des 1. Dias
  bool soundIsPlaying = player->isPlaying();
  if ( soundIsPlaying )
    soundPos = player->getPos();

  // Wann, laut Ablaufplan, soll dieses Dia angezeigt werden:
  int abstime = currSlide.absTime();
  if ( properties->extSlideChange() )
    abstime -= currSlide.duration()/2;
  else
    abstime += currSlide.slideChangeOffset();

  // passe Progress-Anzeige an:
  view->updateActTime(abstime);
  view->updateSlider(abstime);

  if ( currSlide.picFilename() != "ende" )
  {
    // Mitten in der Praesentation
    // ---------------------------

    if ( properties->extSlideChange() )
    {
      // bei einem 'erweitertem Diawechsel' (z.B. Ueberblendung) wird der
      // Timer timerCombineSlide gestartet, der diesen Wechsel in
      // blendingTime durchfuehrt.

      // *** FIXME: ToDo

    } // END if
    
    int duration = currSlide.duration();

    // ueberpruefe nach festgelegter Anzahl an Diawechsel oder nach
    // unzureichender vorhergehender Anpassung auf Verspaetung, und lasse bei
    // Verspaetung die Darstellungszeit dieses Dias anpassen.
    if ( !adjustOk )
    {
#ifndef NDEBUG
      int oldDuration = duration;
#endif
      if ( soundIsPlaying && adjustDuration(duration, adjustOk) )
      {
#ifndef NDEBUG
        cerr << "*** Anpassung ! ("<< currSlide.picFilename() << ") von "
             << oldDuration << " nach " << duration << endl;
#endif
      } // END if adjust

    } // END if

    // starte Timer fuer das naechste Dia. Die Zeit ergibt sich aus der
    // Ruecknahme der Verschiebung des letzten Dias, die Darstelllungsdauer
    // dieses Dias (aus dem Ablaufplan) und ggf. der Verschiebung der
    // Darstellungszeit dieses Dias:
    int realDuration = -lastOffset + duration;
    if ( properties->extSlideChange() )
      realDuration -= (currSlide.blendingTime()/2);
    else
      realDuration += currSlide.slideChangeOffset();

    timerSlide->start(realDuration, true);

    // lade neue Grafik:
    view->loadImage(currSlide.picFilename());

    // packe naechstes Image in den Cache:
    cacheNextImage();
  }
  else
  {
    // Ende der Praesentation erreicht
    // -------------------------------

    // wechsle ggf. vom Fullscreen in den Window Modus
    if ( view->viewMode() == KTourPresenterView::Fullscreen )
      this->slotToggleFullscreen();

    // Setze Schieber und aktuelle Zeit zurueck:
    view->updateSlider(0);
    view->updateActTime(0);

    // zeige Default-Grafik:
    QString defImg = properties->base() + properties->defaultImageFilename();
    view->loadImage(defImg.data());
    view->showImage();
    view->cacheImage(doc->schedule().slide(1).picFilename());

    // setze Ablaufplan zurueck:
    doc->reset();

    status = stopped;
    updateActions();

    // soll versucht werden, den Bildschirmschoner wieder einzuschalten ?
    if ( hasScreensaver )
      ScreensaverControl::setEnabled(true);

    // soll vielleicht das Programm nun beendet werden ?
    if ( properties->autoQuit() )
      close();

    slotStatusMsg(i18n("Ready."));

  } // END if

} // END slotChangeSlide()


bool KTourPresenterApp::adjustDuration(int& duration, bool& adjustOk)
{
  // Strategie, siehe Header-Datei

  // Wenn die Sound-Ausgabe nicht laeuft, ist nichts anzupassen
  if ( !player->isPlaying() )
  {
#ifndef NDEBUG
    cerr << "KTourPresenterApp::adjustDuration(); "
         << "Sound spielt nicht !" << endl;
#endif
    adjustOk = true;
    duration = doc->slide().duration();
    return false;
  } // END if

  // wie verspaetet bin ich :) Zu beachten ist, dass die Sound-Ausgabe erst
  // mit dem ersten Dia startet !
  int lateness = soundPos - doc->slide().absTime();

  if ( abs(lateness) > MAXLATENESS )
  {
    // zu grosse Verspaetung !

    // pruefe, ob eine ggf. noetige Verminderung der Diazeit MINDURATION
    // missachtet
    if ( (duration - lateness) >= MINDURATION )
    {
      // Ok, Anpassung ist ausreichend
      duration -= lateness;
      adjustOk = true;
    }
    else
    {
      // Anpassung reicht nicht !
      duration = MINDURATION;
      adjustOk = false;
    } // END if
  }
  else
  {
    // keine (zu grosse) Verspaetung, nicht unbedingt puenktlich, aber in der
    // Toleranz
    adjustOk = true;

    return false;
  } // END if

  return true;

} // END adjustDuration()


void KTourPresenterApp::cacheNextImage()
{
  if ( !doc->isLastSlide() )
  {
    doc->next();
    view->cacheImage(doc->slide().picFilename());
    doc->prev();
  } // END if
} // END cacheNextImage()


void KTourPresenterApp::slotToggleFullscreen()
{
  if ( status != none )
  {
    view->slotToggleFullscreen();

    if ( view->viewMode() == KTourPresenterView::Window )
    {
      controlFullscreen->setIcon("window_fullscreen");
      controlFullscreen->setText(i18n("Enable &fullscreen"));

      cmenu->changeItem(ID_CONTEXT_FULLSCR,
                        SmallIconSet("window_fullscreen"),
                        i18n("Enable &fullscreen"));
    }
    else
    {
      controlFullscreen->setIcon("window_nofullscreen");
      controlFullscreen->setText(i18n("Disable &fullscreen"));

      cmenu->changeItem(ID_CONTEXT_FULLSCR,
                        SmallIconSet("window_nofullscreen"),
                        i18n("Disable &fullscreen"));
    } // END if viewMode

  } // END if status != none

} // END slotToggleFullscreen()


void KTourPresenterApp::slotPrevDay()
{
  unsigned int newPicIdx;

  doc->prevDay(newPicIdx);
  slotSeek(doc->slide(newPicIdx).absTime());

} // END slotPrevDay()


void KTourPresenterApp::slotNextDay()
{
  unsigned int newPicIdx;

  if ( doc->nextDay(newPicIdx) )
    slotSeek(doc->slide(newPicIdx).absTime());

} // END slotNextDay()


void KTourPresenterApp::slotPrevSlide()
{
  unsigned int slideIdx = doc->slideIdx();

  if ( slideIdx > 0 )
    slotSeek(doc->slide(slideIdx-1).absTime());

} // END slotPrevSlide()


void KTourPresenterApp::slotNextSlide()
{
  unsigned int slideIdx = doc->slideIdx();

  if ( slideIdx < (doc->schedule().size()-2) )
    slotSeek(doc->slide(slideIdx+1).absTime());

} // END slotNextSlide()


void KTourPresenterApp::slotToolbarChanged()
{
  toolBar("mainToolBar")->hide();

  if ( viewToolBar->isChecked() )
    toolBar("mainToolBar")->show();

} // END slotToolbarChanged()


void KTourPresenterApp::slotDisableScreensaver(bool flag)
{
  ScreensaverControl::setEnabled(!flag);
} // END slotDisableScreensaver()


void KTourPresenterApp::slotNewToolbarConfig()
{
  applyMainWindowSettings(KGlobal::config()); //, "MainWindow");
  createGUI();
} // END slotNewToolbarConfig()
