/***************************************************************************
    ktourpresenter.h  -  Die Hauptklasse der Anwendung
    ----------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.38 $
    CVS-ID    : $Id: ktourpresenter.h,v 1.38 2002/02/14 12:53:26 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef KTOURPRESENTER_H
#define KTOURPRESENTER_H


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// STL include files:
#include <exception>

// Qt include files:
#include <qcstring.h>
#include <qevent.h>
#include <qlist.h>
#include <qtimer.h>
#include <qtooltip.h>

// KDE include files:
#include <kaccel.h>
#include <kaction.h>
#include <kapp.h>
#include <kcombobox.h>
#include <kled.h>
#include <klocale.h>
#include <kmainwindow.h>
#include <kpopupmenu.h>
#include <dcopclient.h>

// Application include files:
#include "properties.h"
#include "slide.h"
#include "soundplayer.h"
#include "startuplogo.h"
#include "snapshot.h"


// Forward-Deklarationen:
class KTourPresenterDoc;
class KTourPresenterView;
class ConfigureDialog;


/**
 * Die Basisklasse fuer das KTourPresenter-Anwendungsfenster. Es richtet das
 * Hauptfenster ein und liest die Konfigurationsdatei ein. Ebenso stellt es
 * die Menuezeile, Werkzeugzeile und Statuszeile dar. Eine Instanz von
 * KTourPresneterView entspricht der zentralen Sicht, und ist mit dem
 * Dokumenten-Objekt verbunden.
 * KTourPresneterApp reimplementiert die Methoden, die KMainWindow zum
 * Session-Managment und zur Benutzung der KAction's liefert.
 * @see KMainWindow
 * @see KApplication
 * @see KConfig
 * @see Snapshot
 *
 * @author Dirk Rosert <dirk@kiste.ping.de>
 * $Revision: 1.38 $
 */
class KTourPresenterApp : public KMainWindow
{
  Q_OBJECT

  friend class KTourPresenterView;
  friend class Snapshot;

public:
  /**
   * Konstruktor von KTourPresenterApp, ruft alle Initialisierungsfunktionen
   * fuer den Aufbau der Applikation auf.
   */
  KTourPresenterApp(const QString& playfilename="",
                    QWidget* parent=0, const char* name=0);

  /** Destruktor */
  ~KTourPresenterApp();
    
  /**
   * Oeffnet die uebergebene Abspieldatei mit der URL @p url.
   */
  void openDocumentFile(const KURL& url=0);
    
  /**
   * Gibt einen Zeiger des aktuellen Dokumentes zurueck, das mit der
   * KTMainWindow-Instanz verbunden ist. Es wird von der View-Klasse fuer den
   * Zugriff der Dokumentenobjekte-Methoden benutzt.
   * @see KTourPresenterDoc
   */	
  KTourPresenterDoc *getDocument() const; 	

  /** Initialisiert die Shortcuts. */
  void initAccel();


public slots:

  /** Oeffne eine Praesentationsdatei. */
  void slotFileOpen();
    
  /** Schliesst die Praesentation. */
  void slotFileClose();
    
  /** Schaltet die Toolbar an/aus. */
  void slotViewToolBar();
    
  /** Schaltet die Statuszeile an/aus. */
  void slotViewStatusBar();
    
  /** Erlaubt das Editieren der Tasten. */
  void slotConfigKeys();

  /** Erlaubt das Konfigurieren der Toolbar. */
  void slotConfigToolbar();

  /** Zeigt den Anwendungs-Konfigurationsdialog. */
  void slotConfigure();

  /** Beginne mit der Diavorfuehrung. */
  void slotControlPlay();
    
  /** Unterbreche die Diavorfuehrung. */
  void slotControlPause();
    
  /** Beende die Diavorfuehrung. */
  void slotControlStop();

  /** Zeige den Tour-Bericht an.*/
  void slotControlReport();

  /** Wechsle zu der Aufloesung, die mit @p newres bezeichnet wird. */
  void slotSelectRes(const QString& newres);
  
  /**
   * Wechsle den Inhalt des Bereiches der Statuszeile, der fuer die Anzeige
   * der aktuelle Aktion zustaendig ist.
   * @param text Der Text, der die aktuelle Aktion beschreibt
   */
  void slotStatusMsg(const QString &text);

  /** Starte die Sound-Ausgabe. */
  void slotPlaySound();
  
  /**
   * 'Spule' hin zur Position der Praesentation, die durch die Zeitangabe
   * in Millisekunden (@p msec) angegeben wird.
   */
  void slotSeek(int msec);
  
  /** Zeige das Popup-Menue an der Position @p pos. */
  void slotShowPopup(const QPoint& pos);

  /**
   * Das letzte Fenster der Anwendung wurde geschlossen, die Anwendung kann
   * beendet werden.
   */
  void slotLastWindowClosed();


signals:

  /** Signalisiert das Schliessen der Praesentationsdatei. */
  void sigClose();


protected: // structures:

  /**
   * Statustyp der Praesentation:
   * <ul>
   *   <li>none:    Keine Praesentation geladen
   *   <li>stopped: (Geladene) Praesentation laeuft nicht
   *   <li>running: Praesentation laeuft
   *   <li>paused:  Ablauf der Praesentation ist angehalten
   * </ul>
   */
  enum PresentationStatus { none=0, stopped, running, paused};


protected: // methods

  /**
   * Sichert die allgemeinen Optionen, wie z.B. die Statusbar-Positionen und
   * den Status, genauso wie die Geometrie in die Konfigurationsdatei.
   */ 	
  void saveOptions();

  /**
   * Liest die allgemeinen Optionen wieder ein und initialisiert alle
   * zugehoerigen Variablen.
   */
  void readOptions();

  /**
   * Initialisiert die KActions der Applikation.
   */
  void initActions();

  /**
   * Initialisiert die Statusbar des Hauptfensters.
   */
  void initStatusBar();

  /**
   * Initialisiert das Dokumentenobjekt des Hauptfensters, das mit dem View
   * in initView() verbunden wird.
   * @see initView();
   */
  void initDocument();

  /**
   * Erzeugt das Haupt-Widget der KTMainWindow-Instanz und benutzt es als
   * View.
   */
  void initView();

  /**
   * Sichert beim Ende der Session fuer jedes geoeffnete Anwendungsfenster
   * deren Eigenschaften.
   * @see KTMainWindow#saveProperties
   */
  virtual void saveProperties(KConfig *_cfg);

  /**
   * Liest die Session-Konfigurationsdatei und stellt den Satus der
   * Applikation wieder her, der durch saveProperties() gesichert wurde.
   * @see KTMainWindow#readProperties
   */
  virtual void readProperties(KConfig *_cfg);

  /** Reimplementiert KMainWindow#queryClose(). */
  virtual bool queryClose();

  /** Enabled/Disabled die Aufloesungs-Combobox. */
  void setComboBoxEnabled(bool);


private: // methods:

  /**
   * Enable/disable Menue-Eintraege und Toolbar-Knoepfe, abhaengig vom
   * Zustand.
   */
  void updateActions();
  
  /** Initialisiere die Imlib. */
  void initImlib();

  /**
   * Diese Methode prueft auf Verspaetung und passt ggf. die Darstellungszeit
   * fuer das kommende Dia an. Diese Anpassung findet nur statt, wenn die
   * Verspaetung den maximalen Wert @p MAXLATENESS ueberschritten hat.
   * @param duration Referenz auf die Darstellungszeit, die in dieser Methode
            angepasst werden kann
   * @param adjustOk reicht diese Anpassung (<tt>true</tt>) oder muss im
   *        naechsten Schritt weiter angepasst werden
   * @return gab es eine Verspaetung ?
   */
  bool adjustDuration(int& duration, bool& adjustOk);

  /**
   * Prueft (per DCOP-Aufruf), ob der Bildschirmschoner eingeschaltet ist.
   * @throws DCOPError DCOP-Aufruf fehlgeschlagen
   */
  bool isScreensaverEnabled() const;

  /** Schalte Bildschirmschoner ein bzw. aus. */
  void enableScreensaver(bool flag);


private: // exceptions:

  /**
   * Diese Ausnahme wird geworfen, wenn es Probleme mit einem DCOP-Aufruf
   * gibt.
   */
  class DCOPError : public exception
  {
    public:
      DCOPError() {}

      virtual const char *what() const throw()
      {
        return i18n("DCOP Error !");
      }

  }; // END DCOPError


private slots:

  /** */
  void slotSlideTimerUpdates();

  /** */
  void slotToggleFullscreen();

  /** */
  void slotPrevDay();

  /** */
  void slotNextDay();

  /** */
  void slotPrevSlide();

  /** */
  void slotNextSlide();

  /** */
  void slotToolbarChanged();

  /** */
  void slotDisableScreensaver(bool);


private: // attributes:

  /** */
  DCOPClient *client;

  bool dcopDone;

  QCString realAppId;

  /** War beim Start der Bildschirmschoner eingeschaltet ? */
  bool hasScreensaver;

  /** Das Konfigurationsobjekt der Anwendung. */
  KConfig *config;

  /**
   * @p view ist das Haupt-Widget mit dem Arbeitsbereich.
   * @see KTourPresenterView
   */
  KTourPresenterView *view;

  /**
   * @p doc representiert das aktuelle <em>Dokument</em> (Abspieldatei), und
   * wird nur einmal Erzeugt (<em>Single Document Interface</em>). Es enthaelt
   * den Ablaufplan der Praesentation.
   * @see KTourPresenterDoc
   */
  KTourPresenterDoc *doc;

  /** */
  static QList<KTourPresenterApp> *windowList;

  /**
   * Der Konfigurations-Dialog.
   * @see ConfigureDialog
   * @see Properties
   */
  ConfigureDialog *configdialog;

  // KAction-Zeiger, um diese ein- oder auszuschalten:
  KAction       *fileOpen;
  KAction       *fileClose;
  KAction       *fileQuit;

  KToggleAction *viewToolBar;
  KToggleAction *viewStatusBar;

  KAction       *controlPlay;         // >
  KAction       *controlPause;        // ||
  KAction       *controlStop;         // #

  KAction       *controlPrevDay;
  KAction       *controlNextDay;
  KAction       *controlPrevSlide;
  KAction       *controlNextSlide;
  KAction       *controlFullscreen;
  KAction       *controlReport;

  /** Timer fuer den Diawechsel */
  QTimer *timerSlide;

  int timerSlideId;

  /** Auswahlbox der moeglichen Aufloesungen. */
  QComboBox *resCombo;

  int resComboIdx;

  /** Status der Praesentation (none, stopped, running, paused) */
  PresentationStatus status;

  /** Liste mit den Aufloesungs-Strings, benutzt in der Combo-Box. */
  QStrList *resList;

  /**
   * Der Sound-Player, der die Sound-Datei (u.A. Begleittext) abspielt.
   * @see SoundPlayer
   */
  SoundPlayer *player;

  /**
   * Das zu zeigende Programm-Logo.
   * @see StartupLogo
   */
  StartupLogo *logo;

  /** Objekt der Applikations-Shortcuts. */
  KAccel *m_accel;

  /** Statusleistenelement zur Darstellung von Meldungen. */
  KStatusBarLabel *m_statusLabel;

  /**
   * Status-LED, symbolisiert durch die Farbe den Status. Ein Tool-Tip und
   * WhatsThis gibt Auskunft ueber die Bedeutung der LED-Farbe.
   */
  KLed *stateLed;

  /**
   * Setzt einen Schnappschuss, der nach <em>depausieren</em> Hilft, die
   * Praesentation weiterzufuehren.
   */
  Snapshot snapshot;

  /**
   * 'Durchlaufzaehler', zaehlt die Diawechsel. Nach RUNTHROUGHSIZE Wechseln
   * soll auf Verspaetung geprueft werden.
   */
  int runthroughCounter;

  /** Kontextmenue. */
  KPopupMenu *cmenu;

  /**
   * War die letzte Anpassung ausreichend ? Oder muss im naechsten Schritt
   * weiter angepasst werden.
   */
  bool adjustOk;

  /** */
  int soundPos;

  // Statusbar-ID der Meldungen:
  static const int ID_STATUS_MSG = 0;
  static const int ID_STATUS_LED = 2;

  /// ID's des Kontext-Menues:
  int ID_CONTEXT_PLAY;
  int ID_CONTEXT_STOP;
  int ID_CONTEXT_PAUSE;
  int ID_CONTEXT_SEP1;
  int ID_CONTEXT_PREVDAY;
  int ID_CONTEXT_PREVSLIDE;
  int ID_CONTEXT_NEXTSLIDE;
  int ID_CONTEXT_NEXTDAY;
  int ID_CONTEXT_SEP2;
  int ID_CONTEXT_REPORT;
  int ID_CONTEXT_FULLSCR;
  int ID_CONTEXT_SEP3;
  int ID_CONTEXT_QUIT;

  /**
   * Maximal tolerierte <em>Verspaetung</em> in msec. Ueberschreitet eine
   * ermittelte Verspaetung diesen Wert, so muss die Darstellungszeit des
   * Dias angepasst werden.
   */
  static const int MAXLATENESS = 500;

  /**
   * Minimale (Rest-) Darstellungszeit eines Dias, die nicht unterschritten
   * werden darf.
   */
  static const int MINDURATION = 1000;

  /**
   * Nach wie vielen Diawechseln soll nach <em>Verspaetung</em> geprueft
   * werden.
   */
  static const int RUNTHROUGHS = 1; // bei jedem Diawechsel

  /**
   * Versionsnummer fuer die Konfigurarionsdatei
   * <tt>$KDEDIR/share/config/ktourpresenterrc</tt>. Sie kann in spaeteren
   * Versionen dieser Anwendung wichtig sein, wenn sich das Format aendert.
   */
  static const int CONFIGFILEVERSION = 1;

};


#endif // KTOURPRESENTER_H
