/***************************************************************************
    ktourpresenter.cpp  -  Die Hauptklasse der Anwendung
    ------------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.56 $
    CVS-ID    : $Id: ktourpresenter.cpp,v 1.56 2002/02/14 14:44:27 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// C include files:
#include <unistd.h>

// STL include files:
#include <iostream>

// Qt include files:
#include <qdatetime.h>
#include <qdir.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qstrlist.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#ifdef index
#undef index
#endif

#ifdef KeyPress
#undef KeyPress
#endif

// KDE include files:
#include <kconfig.h>
#include <kcursor.h>
#include <kdatastream.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kkeydialog.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <krun.h>
#include <kstdaction.h>

// Application include files:
#include "ktourpresenter.h"
#include "ktourpresenterview.h"
#include "ktourpresenterdoc.h"
#include "properties.h"
#include "configuredialog.h"
#include "schedule.h"   // fuer Exceptions


// Liste der Applikationsfenster:
QList<KTourPresenterApp> *KTourPresenterApp::windowList = 0L;

// erzeugt in main.cpp:
extern Properties *properties;


KTourPresenterApp::KTourPresenterApp(const QString& playfilename,
                                     QWidget *, const char *name) :
  KMainWindow(0L, name,
              WType_TopLevel | /*WDestructiveClose |*/ WStyle_ContextHelp)
{
  // wenn das letzte Anwendungsfenster geschlossen wurde, beende Anwendung:
  connect(kapp, SIGNAL(lastWindowClosed()),
          this, SLOT(slotLastWindowClosed()));

  bool showLogo = false;

  // kein Sound Player am Anfang
  player = 0L;

  // kein Konfigurationsdialog bisher
  configdialog = 0L;

  // Zaehler der Diawechsel, nach RUNTHOUGHSIZE Wechseln wird auf Verspaetung
  // geprueft und ggf. durch adjustDuration() angepasst
  runthroughCounter = 0;

  // Am Anfang alles im Plan...
  adjustOk = true;

  // erzeuge ggf. die Fensterliste:
  if ( !windowList )
  {
    windowList = new QList<KTourPresenterApp>;
    windowList->setAutoDelete(false);
  } // END if

  // fuege dieses Applikationsfenster ans Ende der Liste an:
  windowList->append(this);

  // ermittle Konfiguration:
  config=kapp->config();

  // beginne mit einem Zustand, in dem (noch) nichts passiert:
  status = none;

  // initialisiere Statusbar, Toolbar, ... und lese die Konfig.-Datei:
  initStatusBar();
  initDocument();
  initView();
  initActions();
  initAccel();
  readOptions();

  // setze die (gespeicherte) Hintergrundfarbe
  view->slotBackgroundColorChanged(properties->backgroundColor());

  // initialisiere die Imlib (fuer Startup-Logo):
  initImlib();

  QString playfile = properties->base() + properties->playFilename();

  // Soll Startup-Logo gezeigt werden ?
  QString pathfile = properties->base()+properties->playFilename();
  bool autoLoadFlag =
    properties->autoLoad() ? QFile::exists(pathfile) : true;
  if ( properties->startupLogo() && // nur wenn Logo angefordert wird, und
       autoLoadFlag              && // nur wenn Abspieldatei existiert, und
       !kapp->isRestored()          // nur wenn App. nicht wiederherstellt
                                    // wurde (Session Management)
     )
  {
    logo = new StartupLogo(0L, "logo");
    showLogo = (logo != 0L);
    if ( showLogo )
    {
      logo->show();
    } // END if showLogo

  } // END if

  // erzeuge String-Liste mit den Aufloeusungs-String fuer die Combo-Box:
  resList = new QStrList();

  // enable/disable Actions
  updateActions();

#if KDE_VERSION >= 220
  // verbinde mit DCOP-Server:
  client = kapp->dcopClient();
  dcopDone = client->attach();
  if ( dcopDone )
    realAppId = client->registerAs(kapp->name());

  // versuche herauszufinden, ob der Bildschirmschoner eingeschaltet ist:
  try
  {
    hasScreensaver = isScreensaverEnabled();
  }
  catch ( DCOPError() )
  {
    dcopDone = false;
  } // END try-catch
#endif

  // Erzeuge Timer fuer das Wechseln der Dias:
  timerSlide = new QTimer(this);
  connect(timerSlide, SIGNAL(timeout()), this, SLOT(slotSlideTimerUpdates()));

  // lade Praesentation am Programmstart ?
  if ( properties->autoLoad() || playfilename != "" )
  {
    setCursor(KCursor::waitCursor());

    if ( playfilename == "" )
      openDocumentFile(playfile);
    else
      openDocumentFile(playfilename);

    setCursor(KCursor::arrowCursor());
  } // END if

  // entferne Startup-Logo (wenn angeszeigt)
  if ( showLogo )
  {
    if ( !properties->autoLoad() )
    {
      // wenn Praesentation nicht geladen werden soll, dann warte etwas,
      // damit das Logo etwas zu sehen ist :)
      sleep(2);  // 2 sec.
    } // END if autoLoad

    logo->hide();
  } // END if showLogo

  // starte ggf. die geladene Praesentation
  if ( properties->autoLoad() && properties->autoStart() )
    slotControlPlay();

} // END KTourPresenterApp()


KTourPresenterApp::~KTourPresenterApp()
{
  windowList->remove(this);

  // ...

} // END ~KTourPresenterApp()


void KTourPresenterApp::initAccel()
{
  // konfigurierbare Shortcuts:
  m_accel = new KAccel(this);

  m_accel->insertItem(i18n("More Brightness"), "More Brightness", "B");
  m_accel->insertItem(i18n("Less Brightness"), "Less Brightness", "SHIFT+B");
  m_accel->insertItem(i18n("More Contrast"),   "More Contrast",   "C");
  m_accel->insertItem(i18n("Less Contrast"),   "Less Contrast",   "SHIFT+C");
  m_accel->insertItem(i18n("More Gamma"),      "More Gamma",      "G");
  m_accel->insertItem(i18n("Less Gamma"),      "Less Gamma",      "SHIFT+G");
  m_accel->insertItem(i18n("Reset Modifier"),    "Reset Modifier",    "R" );
  m_accel->insertItem(i18n("Play Presentation"), "Play Presentation", "P");
  m_accel->insertItem(i18n("Halt Presentation"), "Halt Presentation", "H");
  m_accel->insertItem(i18n("Stop Presentation"), "Stop Presentation", "S");
  m_accel->insertItem(i18n("Start of this day or previous day"),
                                                 "Previous Day",      "Up");
  m_accel->insertItem(i18n("Next day"),          "Next Day",          "Down");
  m_accel->insertItem(i18n("Previous Slide"),    "Previous Slide",    "Left");
  m_accel->insertItem(i18n("Next Slide"),        "Next Slide",       "Right");
  m_accel->insertItem(i18n("Show Tour Report"),  "Show Tour Report",  "F2");
  m_accel->insertItem(i18n("Toggle Fullscreen mode"), "Toggle Fullscreen",
                                                                   "RETURN");

  m_accel->connectItem("More Brightness",   view, SLOT(slotMoreBrightness()));
  m_accel->connectItem("Less Brightness",   view, SLOT(slotLessBrightness()));
  m_accel->connectItem("More Contrast",     view, SLOT(slotMoreContrast()));
  m_accel->connectItem("Less Contrast",     view, SLOT(slotLessContrast()));
  m_accel->connectItem("More Gamma",        view, SLOT(slotMoreGamma()));
  m_accel->connectItem("Less Gamma",        view, SLOT(slotLessGamma()));
  m_accel->connectItem("Reset Modifier",    view, SLOT(slotResetModifier()));
  m_accel->connectItem("Play Presentation", this, SLOT(slotControlPlay()));
  m_accel->connectItem("Halt Presentation", this, SLOT(slotControlPause()));
  m_accel->connectItem("Stop Presentation", this, SLOT(slotControlStop()));
  m_accel->connectItem("Previous Day",      this, SLOT(slotPrevDay()));
  m_accel->connectItem("Next Day",          this, SLOT(slotNextDay()));
  m_accel->connectItem("Previous Slide",    this, SLOT(slotPrevSlide()));
  m_accel->connectItem("Next Slide",        this, SLOT(slotNextSlide()));
  m_accel->connectItem("Show Tour Report",  this, SLOT(slotControlReport()));
  m_accel->connectItem("Toggle Fullscreen", this,
                                            SLOT(slotToggleFullscreen()));

  m_accel->readSettings();

} // END initAccel()


void KTourPresenterApp::initActions()
{
  fileOpen =
    KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());
  fileClose =
    KStdAction::close(this, SLOT(slotFileClose()), actionCollection());
  fileQuit = KStdAction::quit(this, SLOT(close()), actionCollection());
  
  viewToolBar = KStdAction::showToolbar(this, SLOT(slotViewToolBar()), 
                                        actionCollection());
  viewStatusBar = KStdAction::showStatusbar(this, SLOT(slotViewStatusBar()), 
                                            actionCollection());
  
  controlStop = new KAction(i18n("S&top"), "player_stop", 0, this,
                       SLOT(slotControlStop()), actionCollection(),
                       "control_stop");
  controlPlay = new KAction(i18n("&Start"), "1rightarrow", 0, this, 
                       SLOT(slotControlPlay()), actionCollection(),
                       "control_play");
  controlPause = new KAction(i18n("&Pause"), "player_pause", 0, this, 
                       SLOT(slotControlPause()), actionCollection(),
                       "control_pause");
  controlPrevDay = new KAction(i18n("Previous Day"), "2leftarrow", 0, this,
                       SLOT(slotPrevDay()), actionCollection(),
                       "control_prevday");
  controlNextDay = new KAction(i18n("Next Day"), "2rightarrow", 0, this,
                       SLOT(slotNextDay()), actionCollection(),
                       "control_nextday");
  controlPrevSlide = new KAction(i18n("Previous Slide"), "back", 0,
                       this, SLOT(slotPrevSlide()), actionCollection(),
                       "control_prevslide");
  controlNextSlide = new KAction(i18n("Next Slide"), "forward", 0,
                       this, SLOT(slotNextSlide()), actionCollection(),
                       "control_nextslide");
  controlFullscreen = new KAction(i18n("Enable &fullscreen"),
                       "window_fullscreen", 0,
                       this, SLOT(slotToggleFullscreen()), actionCollection(),
                       "control_fullscreen");
  controlReport = new KAction(i18n("Show tour &report"), "package_editors", 0,
                       this, SLOT(slotControlReport()), actionCollection(),
                       "control_report");

  KStdAction::keyBindings(this, SLOT(slotConfigKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(slotConfigToolbar()),
                                actionCollection());
  KStdAction::preferences(this, SLOT(slotConfigure()), actionCollection());
  
  fileOpen->setStatusText(i18n("Opens an existing tour presentation"));
  fileClose->setStatusText(i18n("Closes the actual tour presentation"));
  fileQuit->setStatusText(i18n("Quits KTourPresenter"));
  
  viewToolBar->setStatusText(i18n("Enables/disables the toolbar"));
  viewStatusBar->setStatusText(i18n("Enables/disables the statusbar"));
  
  controlPlay->setStatusText(i18n("Start presentation"));
  controlStop->setStatusText(i18n("Stop presentation"));
  controlPause->setStatusText(i18n("Pause presentation"));
  controlReport->setStatusText(i18n("Show tour report"));
  controlFullscreen->setStatusText(i18n("Enable/disable fullscreen mode"));

  // Kontext-Menue:
  cmenu = new KPopupMenu(0, "contextmenu");
  ID_CONTEXT_PLAY =
    cmenu->insertItem(SmallIconSet("1rightarrow"), i18n("Start presentation"),
                      this, SLOT(slotControlPlay()));
  ID_CONTEXT_PAUSE =
    cmenu->insertItem(SmallIconSet("player_pause"), i18n("Pause presentation"),
                      this, SLOT(slotControlPause()));
  ID_CONTEXT_STOP =
    cmenu->insertItem(SmallIconSet("player_stop"), i18n("Stop presentation"),
                      this, SLOT(slotControlStop()));
  ID_CONTEXT_SEP1 = cmenu->insertSeparator();
  ID_CONTEXT_NEXTDAY =
    cmenu->insertItem(SmallIconSet("2rightarrow"), i18n("Next day"),
                      this, SLOT(slotNextDay()));
  ID_CONTEXT_NEXTSLIDE =
    cmenu->insertItem(SmallIconSet("forward"), i18n("Next slide"),
                      this, SLOT(slotNextSlide()));
  ID_CONTEXT_PREVSLIDE =
    cmenu->insertItem(SmallIconSet("back"), i18n("Previous slide"),
                      this, SLOT(slotPrevSlide()));
  ID_CONTEXT_PREVDAY =
    cmenu->insertItem(SmallIconSet("2leftarrow"), i18n("Previous day"),
                      this, SLOT(slotPrevDay()));
  ID_CONTEXT_SEP2 = cmenu->insertSeparator();
  ID_CONTEXT_REPORT =
    cmenu->insertItem(SmallIconSet("package_editors"),
                      i18n("Show tour report"),
                      this, SLOT(slotControlReport()));
  ID_CONTEXT_FULLSCR =
    cmenu->insertItem(SmallIconSet("window_fullscreen"),
                      i18n("Enable &fullscreen"),
                      this, SLOT(slotToggleFullscreen()));
  ID_CONTEXT_SEP3 = cmenu->insertSeparator();
  ID_CONTEXT_QUIT =
    cmenu->insertItem(SmallIconSet("exit"), i18n("E&xit"),
                      this, SLOT(close()));

  createGUI();
} // END initActions()


void KTourPresenterApp::initStatusBar()
{
  // Meldungen:
  m_statusLabel = new KStatusBarLabel("", ID_STATUS_MSG, statusBar(), "msg");
  m_statusLabel->setFixedHeight(m_statusLabel->sizeHint().height());
  m_statusLabel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
  m_statusLabel->setMargin(1);
  m_statusLabel->setLineWidth(0);
  m_statusLabel->setAlignment(AlignLeft | AlignVCenter);

  // Status-LED:
  stateLed = new KLed(QColor("red"), KLed::Off, KLed::Sunken, KLed::Circular,
                      statusBar(), "led");
  stateLed->setFixedHeight(m_statusLabel->height());
  stateLed->setFixedWidth(m_statusLabel->height());

  // Aufloesungs-Combobox:
  resCombo = new QComboBox(false, statusBar(), "rescombo");
  resCombo->setMinimumWidth(150); // *** FIXME: ugly <<<
  connect(resCombo, SIGNAL(activated(const QString&)),
          this,     SLOT(slotSelectRes(const QString&)));

  statusBar()->addWidget(m_statusLabel, 1, false);
  statusBar()->addWidget(stateLed,      0, true);
  statusBar()->addWidget(resCombo,      0, true);

  QString stateWT = "Status of the resentation.\n"
                    "Stopped presentation (red), "
                    "running presentation (green) and "
                    "paused presentation (yellow).";
  QWhatsThis::add(stateLed, i18n(stateWT.data()));

  QToolTip::add(resCombo, i18n("Resolution"));
  QWhatsThis::add(resCombo,
    i18n("Select a resolution for the images of the presentation."));

} // END initStatusBar()


void KTourPresenterApp::initDocument()
{
  doc = new KTourPresenterDoc(this);
  doc->newDocument();
} // END initDocument()


void KTourPresenterApp::initView()
{ 
  view = new KTourPresenterView(this);
  setCentralWidget(view);	
  doc->addView(view);

  // Praesentationsnamen als Fenstertitel:
  setCaption(doc->presentationName(), false);

  // wenn der Slide bewegt wird
  connect(view, SIGNAL(sigNewSliderPos(int)), this, SLOT(slotSeek(int)));

  //
  connect(this, SIGNAL(sigClose()), view, SLOT(slotReset()));

} // END initView()


void KTourPresenterApp::openDocumentFile(const KURL& url)
{
  QString urlpath = url.path();
  QString message = i18n("Opening file") + " " + urlpath;
  slotStatusMsg(message);

  // separiere aus vollstaendigen Dateinamen den Pfad fuer
  // Properties::base(), und setze diesen:
  int idx = urlpath.findRev('/');
  if ( idx == -1 )
  {
    properties->base("/");
  }
  else
  {
    properties->base(urlpath.left(idx));
    properties->playFilename(urlpath.right(urlpath.length() - idx - 1));
  } // END if

  if ( !url.isEmpty() && doc->openDocument(KURL(urlpath)) )
  {
    setCaption(doc->presentationName(), false);

    // Zeige die Default-Grafik der Praesentation, wenn existient:
    QString startIMG = properties->base()+properties->defaultImageFilename();
    if ( QFile::exists(startIMG) )
    {
      view->loadImage(startIMG.data());
      view->showImage();
    } // END if QFile::exist()

    // Praesentation geladen, aber nicht gestartet:
    status = stopped;

    // enable/disable Actions:
    updateActions();
    
    // Setze die Aufloesungs-Combo:
    doc->resStrList( *resList );
    resCombo->insertStrList(resList);

    // Zeige die Gesamtzeit der Praesentation:
    view->updateTotalTime(doc->totalTime());
    
    // Setzte den Maximalwert fuer den Schieber:
    view->setSliderMax(doc->totalTime()-1);
    
    // Eine Datei wurde geoeffnet, schalte den Fortschrittsschieber ein:
    view->setProgressEnabled(true);

    // Erzeuge ggf. den Sound-Player:
    if ( !player )
      player = new SoundPlayer();

    // und oeffne die Sound-Datei:
    player->openFile(doc->schedule().soundFilename());
    player->play(); player->seek(1); player->pause();  // wirklich noetig :-/

  } // END if !url.isEmpty() && doc->openDocument()
  
  slotStatusMsg(i18n("Ready."));
} // END openDocumentFile()


KTourPresenterDoc *KTourPresenterApp::getDocument() const
{
  return doc;
} // getDocument()


void KTourPresenterApp::saveOptions()
{	
  // [General Options]
  config->setGroup("General Options");
  config->writeEntry("Geometry",     size());
  config->writeEntry("Show Main Toolbar",
                                     viewToolBar->isChecked());
  config->writeEntry("Show Statusbar",
                                     viewStatusBar->isChecked());
  config->writeEntry("Main Toolbar Position",
                                     (int)toolBar("mainToolBar")->barPos());
  config->writeEntry("Main Toolbar Icon Text",
                                     (int)toolBar("mainToolBar")->iconText());
  config->writeEntry("Show Logo",    properties->startupLogo());
  config->writeEntry("Disable Screensaver",
                                     properties->disableScreensaver());

  // [Slideshow Parameters]
  config->setGroup("Slideshow Parameters");
  config->writeEntry("Auto Load",        properties->autoLoad());
  config->writeEntry("Auto Start",       properties->autoStart());
  config->writeEntry("Auto Quit",        properties->autoQuit());
  config->writeEntry("Background Color", properties->backgroundColor());
  config->writeEntry("Fullscreen",       properties->fullscreen());
  config->writeEntry("Hide Mouse",       properties->hideMouse());
  config->writeEntry("Hide Delay",       properties->hideDelay());

  // [File and Directory Names]
  config->setGroup("File and Directory Names");
  config->writeEntry("Base Path",        properties->base());
  config->writeEntry("Play File",        properties->playFilename());
  config->writeEntry("Default Image",    properties->defaultImageFilename());
  config->writeEntry("Report",           properties->reportFilename());

  // [Imlib Parameters]
  config->setGroup("Imlib Parameters");
  config->writeEntry("Own Palette",      properties->ownPalette());
  config->writeEntry("Fast Remap",       properties->fastRemap());
  config->writeEntry("Fast Render",      properties->fastRender());
  config->writeEntry("Dither 16bit",     properties->dither16bit());
  config->writeEntry("Dither 8bit",      properties->dither8bit());
  config->writeEntry("Gamma Value",      properties->gamma());
  config->writeEntry("Brightness Value", properties->brightness());
  config->writeEntry("Contrast Value",   properties->contrast());
  config->writeEntry("Gamma Factor",     properties->gammaFactor());
  config->writeEntry("Brightness Factor",properties->brightnessFactor());
  config->writeEntry("Contrast Factor",  properties->contrastFactor());
  config->writeEntry("Max Cache",        properties->maxCache());
  config->writeEntry("Up Scale",         properties->upScale());
  config->writeEntry("Down Scale",       properties->downScale());
  config->writeEntry("Max Up Scale",     properties->maxUpScale());

  // [Keys]
  m_accel->writeSettings(kapp->config());

  config->sync();
} // END saveOpetions()


void KTourPresenterApp::readOptions()
{
	// [General Options]
  config->setGroup("General Options");

  // config version
  int configVersion = config->readNumEntry("Configuration File Version",
                                           CONFIGFILEVERSION);
  // Dummy-Vergleich; Wichtig vielleicht erst in spaeteren Versionen, wenn
  // sich das Format der Konfig.-Datei aendert; siehe ktourpresenter.h,
  // unten.
  if ( configVersion < 1 )
  { } else { } // END if

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Main Toolbar", true);
  viewToolBar->setChecked(bViewToolbar);
  slotViewToolBar();

  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewStatusBar->setChecked(bViewStatusbar);
  slotViewStatusBar();

  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos =
    (KToolBar::BarPosition) config->readNumEntry("Main Toolbar Position",
                                                 KToolBar::Top);
  toolBar("mainToolBar")->setBarPos(toolBarPos);

  // bar icon text
  KToolBar::IconText iconText;
  iconText =
    (KToolBar::IconText) config->readNumEntry("Main Toolbar Icon Text",
                                                 KToolBar::IconOnly);
  toolBar("mainToolBar")->setIconText(iconText);

  // Geometry
  QSize size=config->readSizeEntry("Geometry");
  if(!size.isEmpty())
    resize(size);

  // show startup logo:
  bool logoFlag = config->readBoolEntry("Show Logo", true);
  properties->startupLogo(logoFlag);

#if KDE_VERSION >= 220
  // disable screensaver:
  bool ssaverFlag = config->readBoolEntry("Disable Screensaver", false);
  properties->disableScreensaver(ssaverFlag);
#else
  properties->disableScreensaver(false);
#endif

  // [Notification Messages]
  config->setGroup("Notification Messages");

  bool confirmExit = config->readBoolEntry("Confirm Exit", true);
  properties->confirmExit(confirmExit);


  // [Slideshow Parameters]
  config->setGroup("Slideshow Parameters");
  
  bool autoLoadFlag = config->readBoolEntry("Auto Load", false);
  properties->autoLoad(autoLoadFlag);
  
  bool autoStartFlag = config->readBoolEntry("Auto Start", false);
  properties->autoStart(autoStartFlag);

  bool autoQuitFlag = config->readBoolEntry("Auto Quit", false);
  properties->autoQuit(autoQuitFlag);

  QColor backgroundC = config->readColorEntry("Background Color",
                                               new QColor("black"));
  properties->backgroundColor(backgroundC);

  bool fullscreenFlag = config->readBoolEntry("Fullscreen", false);
  properties->fullscreen(fullscreenFlag);

  bool hideMouseFlag = config->readBoolEntry("Hide Mouse", false);
  properties->hideMouse(hideMouseFlag);

  int hideDelay = config->readNumEntry("Hide Delay", 3000);
  properties->hideDelay(hideDelay);


  // [File and Directory Names]
  config->setGroup("File and Directory Names");

  QString base = config->readEntry("Base Path", QDir::homeDirPath());
  properties->base(base);
  
  QString playfile = config->readEntry("Play File", "play.ini");
  properties->playFilename(playfile);

  QString defaultImg = config->readEntry("Default Image", "default.jpg");
  properties->defaultImageFilename(defaultImg);
  
  QString report = config->readEntry("Report", "bericht.htm");
  properties->reportFilename(report);


  // [Imlib Parameters]
  config->setGroup("Imlib Parameters");

  bool ownPalette = config->readBoolEntry("Own Palette", true);
  properties->ownPalette(ownPalette);

  bool fastRemap = config->readBoolEntry("Fast Remap", true);
  properties->fastRemap(fastRemap);

  bool fastRender = config->readBoolEntry("Fast Render", true);
  properties->fastRender(fastRender);

  bool dither16 = config->readBoolEntry("Dither 16bit", false);
  properties->dither8bit(dither16);

  bool dither8 = config->readBoolEntry("Dither 8bit", true);
  properties->dither8bit(dither8);

  int gamma = config->readNumEntry("Gamma Value", 0);
  properties->gamma(gamma);

  int brightness = config->readNumEntry("Brightness Value", 0);
  properties->brightness(brightness);

  int contrast = config->readNumEntry("Contrast Value", 0);
  properties->contrast(contrast);

  unsigned int gammaFactor =
    config->readUnsignedNumEntry("Gamma Factor", 10);
  properties->gammaFactor(gammaFactor);

  unsigned int brightnessFactor =
    config->readUnsignedNumEntry("Brightness Factor", 10);
  properties->brightnessFactor(brightnessFactor);

  unsigned int contrastFactor =
    config->readUnsignedNumEntry("Contrast Factor", 10);
  properties->contrastFactor(contrastFactor);

  unsigned int maxCache = config->readUnsignedNumEntry("Max Cache", 32768);
  properties->maxCache(maxCache);

  bool upScale = config->readBoolEntry("Up Scale", false);
  properties->upScale(upScale);

  bool downScale = config->readBoolEntry("Down Scale", true);
  properties->downScale(downScale);

  int maxUpScale = config->readNumEntry("Max Up Scale", 3);
  properties->maxUpScale(maxUpScale);

  // [Keys]
  m_accel->readSettings(kapp->config());

} // END readOptions()


void KTourPresenterApp::saveProperties(KConfig *_cfg)
{
  KURL url=doc->URL();	
  _cfg->writeEntry("filename", url.url());
  QString tempname = kapp->tempSaveName(url.url());
  QString tempurl= KURL::encode_string(tempname);
  //KURL _url(tempurl);
  
  /* *** FIXME: more saved properties !!!
                - if needed the mount device with the presentation on it
                  (e.g. /cdrom)
                - name of the primary config file (e.g. play.ini)
                - choosen resolution or the special config file (e.g.
                  mittel.ply)
                - position in the presentation
   */
} // END saveProperties()


void KTourPresenterApp::readProperties(KConfig* _cfg)
{
  QString filename = _cfg->readEntry("filename", "");
  KURL url(filename);
  
  if ( !filename.isEmpty() )
  {
    doc->openDocument(url);
    setCaption(doc->presentationName(), false);
  }
  
  // *** FIXME: see saveProperties() !!!
  
} // END readProperties()


bool KTourPresenterApp::queryClose()
{
  slotStatusMsg(i18n("Exiting..."));

  bool confirmFlag = properties->confirmExit();
  QString text = i18n("Do you really want to quit this application ?");
  QString caption = i18n("Quit");
  QString button  = i18n("Quit");

#if KDE_VERSION >= 220
  QString dontAskAgainName = "Confirm Exit";
  if ( ( !confirmFlag ) ||
       ( confirmFlag && KMessageBox::warningContinueCancel(0, text,
                      caption, button, dontAskAgainName, true)
                    == KMessageBox::Continue )
     )
#else
  if ( ( !confirmFlag ) ||
       ( confirmFlag && KMessageBox::warningContinueCancel(0, text, caption,
                   button, true) == KMessageBox::Continue )
     )
#endif
  {

#if KDE_VERSION >= 220
    // wenn Bildschirmschoner (immernoch) ausgeschaltet ist, dann mache dies
    // rueckgaengig:
    if ( dcopDone && hasScreensaver )
      enableScreensaver(true);
#endif

    return true;
  } // END if

  slotStatusMsg(i18n("Ready."));

  return false;

} // END queryClose()


void KTourPresenterApp::setComboBoxEnabled(bool flag)
{
  resCombo->setEnabled(flag);
} // END setComboBoxEnabled()


void KTourPresenterApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));
	
  QString dir = QString::null;
  
  if ( QDir(properties->base()).exists() )
  {
    dir = properties->base();
  }

  KURL url=KFileDialog::getOpenURL(dir,
        i18n("play.ini|Play File (play.ini)\n*|All Files"), this,
        i18n("Open File..."));

  try
  {
    if ( status != none )
    {
      player->closeFile();
      doc->deleteContents();
    } // END if

    openDocumentFile(url);
  }
  catch (Schedule::ContentError())
  {
    // *** FIXME: Fehlerbehandlung !!!
  }
  catch (Schedule::CanNotOpenFile(const QString&))
  {
    // *** FIXME: Fehlerbehandlung !!!
  } // END try-catch

  status = stopped;

  // starte ggf. die geladene Praesentation:
  if ( properties->autoStart() )
    slotControlPlay();
  else
    slotStatusMsg(i18n("Ready."));

  setComboBoxEnabled(true);

} // END slotFileOpen()


void KTourPresenterApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));

  // schliesse das Dokument	
  doc->closeDocument();
  player->closeFile();
  emit sigClose();
  status = none;

  setCaption("KTourPresenter");

  updateActions();
  
  setComboBoxEnabled(false);

  slotStatusMsg(i18n("Ready."));

} // END slotFileClose()


void KTourPresenterApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggling toolbar..."));
  
  if ( !viewToolBar->isChecked() )
    toolBar("mainToolBar")->hide();
  else
    toolBar("mainToolBar")->show();

  slotStatusMsg(i18n("Ready."));

} // END slotViewToolBar()


void KTourPresenterApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  
  if ( !viewStatusBar->isChecked() )
    statusBar()->hide();
  else
    statusBar()->show();

  slotStatusMsg(i18n("Ready."));

} // END slotViewStatusBar()


void KTourPresenterApp::slotStatusMsg(const QString& text)
{
  statusBar()->clear();
  m_statusLabel->setText(text);
} // END slotStatusMsg()


void KTourPresenterApp::slotControlPlay()
{
  switch ( status )
  {
    case stopped:
    {
      // Praesentation beginnt von vorne
      // -------------------------------

      // soll im Vollbildmodus gestartet werden ?
      if ( properties->fullscreen() &&
           view->viewMode() == KTourPresenterView::Window )
      {
        this->slotToggleFullscreen();
      } // END if

#if KDE_VERSION >= 220
      // soll versucht werden, den Bildschirmschoner auszuschalten ?
      if ( dcopDone && hasScreensaver && properties->disableScreensaver() )
        enableScreensaver(false);
#endif

#if KDE_VERSION >= 220
      // soll der Mauszeiger nach bestimmter Zeit 'verschwinden' ?
      view->setAutoHideEnabled(properties->hideMouse());
#endif

      doc->reset();

      // eine Aenderung der Ausfloesung ist beim Abspielen der Praesentation
      // nicht moeglich !
      setComboBoxEnabled(false);

      // beginne mit dem Abspielen der Sound-Datei:
      slotPlaySound();

      // starte Timer fuer das erste Dia:
      timerSlide->start(doc->slide().duration(), true);

      // lade erste Grafik (2. Stelle im Ablaufplan) in den Cache:
      view->cacheImage(doc->schedule().slide(1).picFilename());

      // loesche einen ggf. vorher erzeugten Schnappschuss
      snapshot.erase();

      // nun 'laeuft' die Praesentation
      status = running;

      break;
    } // case running

  case paused:
    {
      // eine Praesentation wurde vorher angehalten
      // ------------------------------------------

      // Stelle den Schnappschuss wieder her
      if ( snapshot.isValid() )
      {
        // Spule die Sound-Datei zur Schnappschusszeit
        slotSeek(snapshot.absTime());

        // fuehre Sound-Ausgabe weiter
        if ( player->isPlaying() || player->isPaused() )
        {
          player->pause();
        }
        else
        {
          // bisher wurde Sound-Ausgabe noch nicht gestartet, Praesentation
          // kam durch Verschieben des Sliders vor dem Start in den Pause-
          // Modus
          player->play();
          player->seek(snapshot.absTime());
        } // END if

        // starte Timer fuer die Restzeit des Dias
        timerSlide->start(snapshot.remainingTime(), true);

        // loesche den vorher erzeugten Schnappschuss
        snapshot.erase();

        // nun 'laeuft' die Praesentation wieder
        status = running;

#if KDE_VERSION >= 220
        // soll der Mauszeiger nach bestimmter Zeit 'verschwinden' ?
        view->setAutoHideEnabled(properties->hideMouse());
#endif

      } // END if

      break;
    } // case paused:

  default:
    break;
  } // END switch

  updateActions();

  slotStatusMsg(i18n("Playing presentation..."));

} // END slotControlPlay()


void KTourPresenterApp::slotControlPause()
{
  switch ( status )
  {
    case running:
      {
        // Setzte Schnappschuss:
        int absTime = player->getPos();  // *** FIXME: was, wenn kein Sound-S.
        snapshot.set(doc, doc->slideIdx(), absTime);

        // halte Sound-Player an
        player->pause();

        // Stoppe Timer fuer Diawechsel
        timerSlide->stop();

        status = paused;

        slotStatusMsg(i18n("Delay presentation."));

#if KDE_VERSION >= 220
        // einen 'verschwundenen' Mauszeiger wieder zeigen
        if ( properties->hideMouse() )
          view->setAutoHideEnabled(false);
#endif

        break;
      } // case running

    case paused:
      {
        slotControlPlay();

        break;
      } // case paused

    default:
    {
      // Kann das vorkommen ?

      break;
    } // default

  } // END switch

  updateActions();

} // END slotControlPause()


void KTourPresenterApp::slotControlStop()
{
  // beende Timer fuer Dia-Wechsel:
  timerSlide->stop();

  // beende Abspielen der Sound-Daten:
  player->stop();

  // wechsle ggf. vom Fullscreen in den Window Modus
  if ( view->viewMode() == KTourPresenterView::Fullscreen )
    this->slotToggleFullscreen();

  // soll versucht werden, den Bildschirmschoner wieder einzuschalten ?
#if KDE_VERSION >= 220
  if ( dcopDone && hasScreensaver )
    enableScreensaver(true);
#endif

#if KDE_VERSION >= 220
  // einen 'verschwundenen' Mauszeiger wieder zeigen
  if ( properties->hideMouse() )
    view->setAutoHideEnabled(false);
#endif

  // Zeige Default-Grafik dieser Praesentation:
  QString defImg = properties->base() + properties->defaultImageFilename();
  view->loadImage(defImg.data());

  // Setze Fortschrittsanzeige zurueck:
  view->updateActTime(0);
  view->updateSlider(0);

  // Eine Wahl der Aufloesung nun wieder moeglich:
  setComboBoxEnabled(true);

  // Setze Dokument zurueck:
  doc->reset();

  view->cacheImage(doc->schedule().slide(1).picFilename());

  // Loesche den Schnappschuss:
  snapshot.erase();

  status = stopped;

  updateActions();

  slotStatusMsg(i18n("Ready."));

} // END slotControlStop()


void KTourPresenterApp::slotControlReport()
{
  QString url = properties->base()+properties->reportFilename();
    
  KRun::run(properties->browserName(), QStringList(url));
} // END slotControlReport()


void KTourPresenterApp::slotConfigKeys()
{
  KKeyDialog::configureKeys(m_accel);
} // END slotConfigKeys()


void KTourPresenterApp::slotConfigToolbar()
{
  KEditToolbar dlg(factory());

  if (dlg.exec())
    createGUI();

} // END slotConfigToolbar()


void KTourPresenterApp::slotConfigure()
{
  if ( configdialog == 0L )
  {
    configdialog = new ConfigureDialog(topLevelWidget(), "configdialog",
                                       m_accel, this, false);
    connect(configdialog, SIGNAL(sigBackgroundColorChanged(const QColor&)),
            view,         SLOT(slotBackgroundColorChanged(const QColor&)));
    connect(configdialog, SIGNAL(sigModifierChanged()),
            view        , SLOT(slotModifierChanged()));
    connect(configdialog, SIGNAL(sigToolbarChanged()),
                          SLOT(slotToolbarChanged()));
    connect(configdialog, SIGNAL(sigDisableScreensaver(bool)),
                          SLOT(slotDisableScreensaver(bool)));
  }
  else
  {
    // Properties haben sich vielleicht geaendert, lasse Eingabe-Elemente
    // mit den aktuellen Werten aus Properties updaten
    configdialog->setValues();
  } // END if

  configdialog->show();

} // END slotConfigure()


void KTourPresenterApp::slotSelectRes(const QString& newres)
{
  // wechsle zur gewaehlten Aufloesung
  doc->setScheduleByName(newres);

  QString imageName;
  if ( status == stopped )
    imageName = properties->base() + properties->defaultImageFilename();
  else
    imageName = doc->slide().picFilename();

  // lade die Grafik der neu gewaehlten Aufloesung
  if ( !properties->hideMouse() )
    setCursor(KCursor::waitCursor());
  view->loadImage(imageName);
  if ( !properties->hideMouse() )
    setCursor(KCursor::arrowCursor());

} // END slotSelectRes()


void KTourPresenterApp::slotPlaySound()
{
  player->play();
} // END slotPlaySound()


void KTourPresenterApp::slotSeek(int msec)
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::waitCursor());

  int seekTime = doc->seek(msec);
  
  // aendere actTimeLabel:
  view->updateActTime(doc->slide().absTime());

  // lade neue Grafik:
  view->loadImage(doc->slide().picFilename());
  view->showImage();
  
  player->seek(seekTime);

  if ( status == running )
  {
    timerSlide->start(doc->slide().duration(), true);
    int slideIdx = doc->schedule().currSlideIdx();
    view->cacheImage(doc->schedule().slide(slideIdx+1).picFilename());
    //slideTimerUpdates();
  } // END if
  else
  {
    if ( (status == stopped) || (status == paused) )
    {
      // das Verschieben des Sliders bei einer nicht gestarteten (stopped)
      // Praesentation laesst den Status in paused wechseln => von der neuen
      // Position muss ein Snapshot gesetzt werden, analoges gilt fuer letztes
      // auch wenn der Zustand schon paused war

#if KDE_VERSION >= 220
      if ( status == stopped )
      {
        // soll versucht werden, den Bildschirmschoner auszuschalten ?
        if ( dcopDone && hasScreensaver && properties->disableScreensaver() )
          enableScreensaver(false);
      } // END if
#endif

      status = paused;

      snapshot.set(doc, doc->slideIdx(), doc->slide().absTime());

      // speichere das naechste Dia ziwschen
      doc->next();
      view->cacheImage(doc->slide().picFilename());
      doc->prev();

      updateActions();
    } // END if
  } // END if

  if ( !properties->hideMouse() )
    setCursor(KCursor::arrowCursor());

} // END slotSeek()


void KTourPresenterApp::slotShowPopup(const QPoint& pos)
{
  cmenu->popup(pos);
} // END slotShowPopup()


void KTourPresenterApp::slotLastWindowClosed()
{
  saveOptions();
} // END slotLastWindowClosed()


void KTourPresenterApp::updateActions()
{
  // exisitert Tourberichts-Datei ?
  bool existReport =
    QFile::exists(properties->base() + properties->reportFilename());
  controlReport->setEnabled(existReport);
  cmenu->setItemEnabled(ID_CONTEXT_REPORT, existReport);

  cmenu->setItemEnabled(ID_CONTEXT_PREVDAY,   true);
  cmenu->setItemEnabled(ID_CONTEXT_NEXTDAY,   true);
  cmenu->setItemEnabled(ID_CONTEXT_PREVSLIDE, true);
  cmenu->setItemEnabled(ID_CONTEXT_NEXTSLIDE, true);
  cmenu->setItemEnabled(ID_CONTEXT_FULLSCR,   true);

  // setze Actions abhaengig vom Status
  switch ( status )
  {
    case running:
      {
        fileOpen->setEnabled(false);
        fileClose->setEnabled(false);

        controlPlay->setEnabled(false);
        controlStop->setEnabled(true);
        controlPause->setEnabled(true);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  false);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  true);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, true);

        setComboBoxEnabled(false);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("green"));
        QToolTip::add(stateLed, i18n("Status: Running"));

        break;
      } // case running

    case paused:
      {
        fileOpen->setEnabled(false);
        fileClose->setEnabled(false);

        controlPlay->setEnabled(true);
        controlStop->setEnabled(true);
        controlPause->setEnabled(true);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  true);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  true);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, true);

        setComboBoxEnabled(true);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("yellow"));
        QToolTip::add(stateLed, i18n("Status: Paused"));

        break;
      } // case paused:

    case stopped:
      {
        fileOpen->setEnabled(true);
        fileClose->setEnabled(true);

        controlPlay->setEnabled(true);
        controlStop->setEnabled(false);
        controlPause->setEnabled(false);
        controlFullscreen->setEnabled(true);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,  true);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,  false);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE, false);

        setComboBoxEnabled(true);

        stateLed->setState(KLed::On);
        stateLed->setColor(QColor("red"));
        QToolTip::add(stateLed, i18n("Status: Stopped"));

        break;
      } // END stopped:

    case none:
      {
        fileOpen->setEnabled(true);
        fileClose->setEnabled(false);

        controlPlay->setEnabled(false);
        controlStop->setEnabled(false);
        controlPause->setEnabled(false);
        controlFullscreen->setEnabled(false);
        controlReport->setEnabled(false);

        cmenu->setItemEnabled(ID_CONTEXT_PLAY,      false);
        cmenu->setItemEnabled(ID_CONTEXT_STOP,      false);
        cmenu->setItemEnabled(ID_CONTEXT_PAUSE,     false);
        cmenu->setItemEnabled(ID_CONTEXT_REPORT,    false);
        cmenu->setItemEnabled(ID_CONTEXT_PREVDAY,   false);
        cmenu->setItemEnabled(ID_CONTEXT_NEXTDAY,   false);
        cmenu->setItemEnabled(ID_CONTEXT_PREVSLIDE, false);
        cmenu->setItemEnabled(ID_CONTEXT_NEXTSLIDE, false);
        cmenu->setItemEnabled(ID_CONTEXT_FULLSCR,   false);

        setComboBoxEnabled(false);

        stateLed->setState(KLed::Off);
        QToolTip::add(stateLed, i18n("Status: No presentation"));

        break;
      } // case none:

    default:
      { break; }

  } // END switch

} // END updateActions()


void KTourPresenterApp::slotSlideTimerUpdates()
{
#ifndef NDEBUG
  cerr << "KTourPresenterApp::slotSlideTimerUpdates()" << endl;
#endif

  soundPos = player->getPos();

  // wechsle nun zum naechsten Dia in der Praesentation:
  doc->next();

  int abstime = doc->slide().absTime();

  // passe actTimeLabel an:
  view->updateActTime(abstime);

  // passe Progress-Slider an:
  view->updateSlider(abstime);

  if ( doc->slide().picFilename() == "ende" )
  {
    // Ende der Praesentation erreicht
    // -------------------------------

    // wechsle ggf. vom Fullscreen in den Window Modus
    if ( view->viewMode() == KTourPresenterView::Fullscreen )
    {
      this->slotToggleFullscreen();
    } //END if

    // Setze Schieber und aktuelle Zeit zurueck:
    view->updateSlider(0);
    view->updateActTime(0);

    // zeige Default-Grafik:
    QString defImg = properties->base() + properties->defaultImageFilename();
    view->loadImage(defImg.data());
    view->showImage();
    view->cacheImage(doc->schedule().slide(1).picFilename());

    // setze Ablaufplan zurueck:
    doc->reset();

    status = stopped;
    updateActions();

#if KDE_VERSION >= 220
    // soll versucht werden, den Bildschirmschoner wieder einzuschalten ?
    if ( dcopDone && hasScreensaver )
      enableScreensaver(true);
#endif

    // soll vielleicht das Programm nun beendet werden ?
    if ( properties->autoQuit() )
      close();

    slotStatusMsg(i18n("Ready."));
  }
  else
  {
    // Mitten in der Praesentation
    // ---------------------------

    // erhoehe 'Durchlaufzaehler'
    runthroughCounter++;

    int duration = doc->slide().duration();

    // ueberpruefe nach festgelegter Anzahl an Diawechsel oder nach
    // unzureichender vorhergehender Anpassung auf Verspaetung, und lasse bei
    // Verspaetung die Darstellungszeit dieses Dias anpassen.
    if ( (runthroughCounter % RUNTHROUGHS == 0) || !adjustOk )
    {
#ifndef NDEBUG
      int oldDuration = duration;
#endif
      if ( adjustDuration(duration, adjustOk) )
      {
#ifndef NDEBUG
        cerr << "*** Anpassung ! ("<< doc->slide().picFilename() << ") von "
             << oldDuration << " nach " << duration << endl;
#endif
      } // END if adjust

    } // END if runthoughCounter

    // starte Timer fuer das naechste Dia:
    timerSlide->start(duration, true);

    // lade neue Grafik:
    view->loadImage(doc->slide().picFilename());

    // packe naechstes Image in den Cache:
    doc->next();
    view->cacheImage(doc->slide().picFilename());
    doc->prev();

  } // END if

} // END slotSlideTimerUpdates()


void KTourPresenterApp::initImlib()
{
  ImlibInitParams par;
  par.flags = ( PARAMS_REMAP | PARAMS_FASTRENDER | PARAMS_HIQUALITY |
       PARAMS_DITHER | PARAMS_IMAGECACHESIZE | PARAMS_PIXMAPCACHESIZE );

  par.paletteoverride   = properties->ownPalette()  ? 1 : 0;
  par.remap             = properties->fastRemap()   ? 1 : 0;
  par.fastrender        = properties->fastRender()  ? 1 : 0;
  par.hiquality         = properties->dither16bit() ? 1 : 0;
  par.dither            = properties->dither8bit()  ? 1 : 0;
  unsigned int maxcache = properties->maxCache();

  // 0 == kein cache
  par.imagecachesize  = maxcache * 1024;
  par.pixmapcachesize = maxcache * 1024;

} // END initImlib()


bool KTourPresenterApp::adjustDuration(int& duration, bool& adjustOk)
{
  // *Strategie*: Massgebend fuer die Verspaetung ist die momentane Zeit beim
  //   Abspielen der Sound-Datei (in msec. seit Anfang der Sound-Datei). Die
  //   Differenz zwischen der abs. Diawechselzeit aus dem Ablaufplan und der
  //   Position in der Sound-Datei ergibt die Verspaetung. Die Diawechsel-
  //   zeit wird nur angepasst, wenn sie MAXLATENESS ueberschreitet (um nicht
  //   schon bei 1 msec. Verspaetung anzupassen). Die Abspielzeit des
  //   Nachfolgedias wird um diesen Wert vermindert. Dabei darf diese
  //   angepasste Abspielzeit den Wert MINDURATION nicht unterschritten
  //   werden. Andernfalls wird (duration - MINDURATION) zurueckgegeben (damit
  //   dieses wenigstens MINDURATION msec. dargestellt werden) und
  //   adjustOk=false gesetzt, um im naechsten Schritt eine weitere Anpassung
  //   durchzufuehren.

  // wie verspaetet bin ich :)
  int lateness = soundPos - doc->slide().absTime();
  if ( abs(lateness) > MAXLATENESS )
  {
    // zu grosse Verspaetung !

    // pruefe, ob eine ggf. noetige Verminderung der Diazeit MINDURATION
    // missachtet
    if ( duration - lateness >= MINDURATION )
    {
      // Ok, Anpassung ist ausreichend
      duration -= lateness;
      adjustOk = true;
    }
    else
    {
      // Anpassung reicht nicht !
      duration = MINDURATION;
      adjustOk = false;
    } // END if
  }
  else
  {
    // keine (zu grosse) Verspaetung, nicht unbedingt puenktlich, aber in der
    // Toleranz
    adjustOk = true;

    return false;
  } // END if

  return true;

} // END adjustDuration()


bool KTourPresenterApp::isScreensaverEnabled() const
{
  QCString data;
  QCString reply_data;
  QCString reply_type;

  if ( !client->call("kdesktop",
                     "KScreensaverIface",
                     "isEnabled()",
                     data,
                     reply_type,
                     reply_data
                    )
     )
  {
    throw DCOPError();
  }
  else
  {
    if ( reply_type != "int" )
    {
      throw DCOPError();
    } // END if
  } // END if

  QDataStream answer(reply_data, IO_ReadOnly);
  int result;
  answer >> result;

  return result != 0;
} // END isScreensaverEndabled()


void KTourPresenterApp::enableScreensaver(bool flag)
{
  QByteArray data;
  QCString reply_type;
  QCString reply_data;

  QDataStream arg(data, IO_WriteOnly);
  arg << flag; //(flag ? 1 : 0);
  if ( !client->call("kdesktop",
                     "KScreensaverIface",
                     "enable(bool)",
                     data,
                     reply_type,
                     reply_data
                    )
     )
  {
    throw DCOPError();
  }
  else
  {
    if ( reply_type != "bool" )
    {
      throw DCOPError();
    } // END if
  } // END if

  QDataStream answer(reply_data, IO_ReadOnly);
  bool result;
  answer >> result;

} // END enableScreensaver()


void KTourPresenterApp::slotToggleFullscreen()
{
  if ( status != none )
  {
    view->slotToggleFullscreen();

    if ( view->viewMode() == KTourPresenterView::Window )
    {
      controlFullscreen->setIcon("window_fullscreen");
      controlFullscreen->setText(i18n("Enable &fullscreen"));

      cmenu->changeItem(ID_CONTEXT_FULLSCR,
                        SmallIconSet("window_fullscreen"),
                        i18n("Enable &fullscreen"));
    }
    else
    {
      controlFullscreen->setIcon("window_nofullscreen");
      controlFullscreen->setText(i18n("Disable &fullscreen"));

      cmenu->changeItem(ID_CONTEXT_FULLSCR,
                        SmallIconSet("window_nofullscreen"),
                        i18n("Disable &fullscreen"));
    } // END if viewMode

  } // END if status != none

} // END slotToggleFullscreen()


void KTourPresenterApp::slotPrevDay()
{
  unsigned int newPicIdx;

  doc->prevDay(newPicIdx);
  slotSeek(doc->slide(newPicIdx).absTime());

} // END slotPrevDay()


void KTourPresenterApp::slotNextDay()
{
  unsigned int newPicIdx;

  if ( doc->nextDay(newPicIdx) )
    slotSeek(doc->slide(newPicIdx).absTime());

} // END slotNextDay()


void KTourPresenterApp::slotPrevSlide()
{
  unsigned int slideIdx = doc->slideIdx();

  if ( slideIdx > 0 )
    slotSeek(doc->slide(slideIdx-1).absTime());

} // END slotPrevSlide()


void KTourPresenterApp::slotNextSlide()
{
  unsigned int slideIdx = doc->slideIdx();

  if ( slideIdx < (doc->schedule().size()-2) )
    slotSeek(doc->slide(slideIdx+1).absTime());

} // END slotNextSlide()


void KTourPresenterApp::slotToolbarChanged()
{
  toolBar("mainToolBar")->hide();

  if ( viewToolBar->isChecked() )
    toolBar("mainToolBar")->show();

} // END slotToolbarChanged()


void KTourPresenterApp::slotDisableScreensaver(bool flag)
{
  cerr << "KTourPresenterApp::slotDisableScreensaver("
       << flag << ")" << endl;
  enableScreensaver(!flag);
} // END slotDisableScreensaver()
