/***************************************************************************
    imagecache.cpp  -  Repraesentiert einen Cache von KTourImage-Objekte
    --------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.15 $
    CVS-ID    : $Id: imagecache.cpp,v 1.15 2002/02/02 19:19:51 dirk Exp $

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * basiert auf KuickShow 0.8.2 von Carsten Pfeiffer <pfeiffer@kde.org>
 */


// C include files:
#include <sys/time.h>
#include <unistd.h>
#include <stdlib.h>

// STL include files:
#include <iostream>

// Qt include files:
#include <qcolor.h>
#include <qfile.h>
#include <qobject.h>
#include <qpalette.h>

// KDE include files:
#include <kcursor.h>

// Application include files:
#include "imagecache.h"
#include "properties.h"


// erzeugt in main.cpp:
extern Properties *properties;


ImageCache::ImageCache(ImlibData *id, int maxImages)
{
  m_id        = id;
  idleCount   = 0;
  m_maxImages = maxImages;
  kuickList.setAutoDelete(true);
  fileList.clear();
  kuickList.clear();
} // END ImageCache()


ImageCache::~ImageCache()
{
  kuickList.clear();
  fileList.clear();
} // END ~ImageCache()


void ImageCache::setMaxImages(int maxImages)
{
  m_maxImages = maxImages;
  int count   = kuickList.count();
  
  while ( count > m_maxImages )
  {
    kuickList.removeLast();
    fileList.remove(fileList.fromLast());
    count--;
  } // END while

} // END setMaxImages()


int ImageCache::maxImages() const
{
  return m_maxImages;
} // END maxImages()


void ImageCache::slotBusy()
{
  if ( idleCount == 0 )
    emit sigBusy();

  idleCount++;
} // END slotBusy()


void ImageCache::slotIdle()
{
  idleCount--;

  if ( idleCount == 0 )
    emit sigIdle();
} // END slotIdle()


KTourImage *ImageCache::getKTimage(const QString& file,
                                   ImlibColorModifier mod, bool paintFlag)
{
  KTourImage *kuim = 0L;
  if ( file.isEmpty() )
    return 0L;

  // ermittle den Index im Cache, an dem sich das Image befindet
  int index = fileList.findIndex( file );

  if ( index != -1 ) // Image ist im Cache !
  {

#ifndef NDEBUG
    cerr << "ImageCache::getKTimage(" << file.latin1() << ") im Cache"
         << endl;
#endif	

    if ( index == 0 )
      kuim = kuickList.at(0);

    // Reorganisation der beiden Listen.
    else
    {
      kuim = kuickList.take(index);
      kuickList.insert(0, kuim);
      fileList.remove(file);
      fileList.prepend(file);
    } // END if

    // wende Modifikatoren an, sie koennen sich seitdem sich die Grafik im
    // Cache befindet, geaendert haben:
    mod.brightness = properties->brightness() + ImlibOffset;
    mod.contrast   = properties->contrast()   + ImlibOffset;
    mod.gamma      = properties->gamma()      + ImlibOffset;
    Imlib_set_image_modifier(kuim->imlibData(), kuim->imlibImage(), &mod);
    //kuim->setDirty(true); // laesst Grafik neuzeichnen

    return kuim;
  }

  if ( !kuim )
  {
    // Image war nicht im Cache !

    slotBusy();

#ifndef NDEBUG
    struct timeval tms1, tms2;
    gettimeofday(&tms1, NULL);
#endif

    ImlibImage *im =
      Imlib_load_image(m_id, QFile::encodeName(file).data());

#ifndef NDEBUG
    gettimeofday(&tms2, NULL);
    int diffTime = (tms2.tv_usec - tms1.tv_usec) / 1000;
    diffTime = (diffTime < 0) ? (diffTime + 1000) : diffTime;
    cerr << "ImageCache::getKTimage(); loading image " << file.latin1()
         << " took " << diffTime << " ms" << endl;
#endif	

    slotIdle();
    if ( !im )
      return 0L;

    Imlib_set_image_modifier(m_id, im, &mod);
    kuim = new KTourImage(file, im, m_id, paintFlag);
    connect(kuim, SIGNAL(startRendering()),   SLOT(slotBusy()));
    connect(kuim, SIGNAL(stoppedRendering()), SLOT(slotIdle()));

    kuickList.insert(0, kuim);
    fileList.prepend(file);
  } // END if !kuim

  if ( kuickList.count() > (unsigned int) m_maxImages )
  {
    kuickList.removeLast();
    fileList.remove(fileList.fromLast());
  } // END if

  return kuim;
} // END getKTimage()
