/***************************************************************************
    schedule.h  -  Ablaufplan einer Praesentation
    ----------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.13 $
    CVS-ID    : $Id: schedule.h,v 1.13 2002/02/11 15:14:37 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef SCHEDULE_H
#define SCHEDULE_H


// C include files:
#include <stdio.h>

// STL include files:
#include <vector>
#include <exception>

// Qt include files:
#include <qobject.h>
#include <qstring.h>

// KDE include file:
#include <klocale.h>  // fuer i18n()

// Application include files:
#include "slide.h"


/**
 * Klasse Schedule
 *
 * Ein Ablaufplan (Schedule) repraesentiert eine Praesentation in einer
 * bestimmten Aufloesung (z.B. 768x512, 1536x1024 oder 3072x2048).
 *
 * @see Slide
 * @author Dirk Rosert <dirk@kiste.ping.de>
 * $Revision: 1.13 $
 */
class Schedule
{
public:

  /**
   * Erzeugt einen neuen Ablaufplan, mit der Ausfloesungsbezeichnung
   * @p resolution und der aufloesungsabhaengigen Abspieldatei
   * @p schedulefilename. Diese Datei wird gelesen. Aus den Informationen
   * wird eine Sequenz von von Slides / Dias gebildet.
   *
   * @throws CanNotOpenFile @p schedulefilename existiert nicht !
   */
  Schedule(const QString& resolution,
	         const QString& schedulefilename);
  
  ~Schedule();

  /** Gibt den String der die Aufloesung beschreibt zurueck. */
  QString resolutionString() const;
  
  /**
   * Gibt den Namen der Praesentations- oder Schedule-Datei incl. Pfad zurueck
   * (z.B. /cdrom/fein.ply).
   */
  QString& scheduleFilename();
  
  /** Gibt den Namen der Sound-Datei zurueck. */
  QString& soundFilename();
  
  /** Gibt die Anzahl der Dias dieser Praesentation zurueck. */
  unsigned int size() const;

  /** Gibt den Index des aktuellen Dias zurueck. */
  int currSlideIdx() const;

  /** Gibt die Dauer der Praesentation in msec. zurueck. */
  int duration() const;
  
  /**
   * Gibt das Dia mit dem Index <tt>idx</tt> in dieser Praesentation zurueck.
   *
   * @see Slide
   * @throws OutOfRange Der Index ist ausserhalb des gueltigen Bereiches. Es
   *         wurde ein Dia ausgewaehlt, dessen Index groesser ist, als der
   *         des letzten Dias.
   */
  Slide& slide(unsigned int idx);

  /** Gibt das aktuelle Dia zurueck. */
  Slide& slide();

  /**
   * Gibt das naechste Dia in @p nextSlide zurueck, wenn es ein naechstes Dia
   * gibt.
   * @return Ok ?
   */
  bool nextSlide(Slide& nextslide);

  /**
   * Geht zum Anfang des Tages, zu dem das Dia mit dem Index @p picIdx
   * gehoert, oder zum vorigen Tag, wenn @picIdx schon der Anfang dieses
   * Tages ist.
   * @param picIdx Index des aktuellen Dias.
   * @param newPicIdx Index des neuen Dias (call by reference!)
   */
  void prevDay(unsigned int picIdx, unsigned int& newPicIdx);

  /**
   * Geht zum naechsten Tag, der nach diesem Tag folgt, zu dem das Dia mit
   * dem Index @p picIdx gehoert.
   * @param picIdx Index des aktuellen Dias.
   * @param newPicIdx Index des neuen Dias (call by reference!)
   * @return Gibt es ueberhaupt einen <em>naechsten Tag</em> (ist somit
   *         @p newPicIdx guelitg ?)
   */
  bool nextDay(unsigned int picIdx, unsigned int& newPicIdx);

  /**
   * Setzt den Ablaufplan zurueck. Das <em>naechste Dia</em> dieser
   * Prasentation ist damit das erste Dia in diesem Ablaufplan.
   */
  void reset();


public: // exceptions:

  /**
   * Diese Ausnahme wird geworfen, wenn die ausgewaehlte Datei nicht geoeffnet
   * werden konnte. Vielleicht weil diese Datei nicht existiert.
   * @see Schedule
   */
  class CanNotOpenFile : public exception
  {
  public:
    /** Wird geworfen, wenn @p filename nicht geladen werden konnte. */
    CanNotOpenFile(const QString& filename) :
      m_filename(filename)
    { }

    /** Gibt die Fehlermeldung zurueck. */
    virtual const char *what() const throw()
    {
      QString result = i18n("Can't open %1 !").arg(m_filename);
      return result.data();
    }

  protected:
    QString m_filename;
  };

  /**
   * Diese Ausnahme wird geworfen, wenn ein Dia (Slide) per Index ausgewaehlt
   * wurde, das nicht existiert (Indexfehler).
   * @see Schedule
   * @see Slide
   */
  class OutOfRange : public exception
  {
  public:
    /**
     * Wird geworfen, wenn ein Dia in dem Ablaufplan angefordert wird, dessen
     * Index ausserhalb der Grenzen liegt.
     */
    OutOfRange() {}

    /** Gibt die Fehlermeldung zurueck. */
    virtual const char *what() const throw()
    {
      return i18n("Out of range !");
    }
  };

  /**
   * Diese Ausnahme wird geworfen, wenn der dem Konstruktor von Schedule
   * uebergebene Ablaufplandatei keine solche ist. Der Inhalt kann nicht
   * als Ablaufplan interpretiert werden.
   * @see Schedule
   */
  class ContentError : public exception
  {
  public:
    /** Wird geworfen, wenn es sich nicht um eine Ablaufplandatei handelt. */
    ContentError(const QString& filename) :
      m_filename(filename)
    { }

    /** Gibt die Fehlermeldung zurueck. */
    virtual const char *what() const throw()
    {
      QString result = m_filename + i18n(" is not a schedule file !");
      return result.data();
    }

  protected:
    QString m_filename;
  };


protected: // methods:

  /** Liest eine Zeile aus der aufloesungsabhaengigen Abpieldatei aus. */
  QString getline(FILE *pFile);

  /** Liefert den Index des Tages, zu dem das Dia @p slideIdx gehoert. */
  unsigned int partOfDay(unsigned int slideIdx) const;


protected: // structures:

  /**
   * Definiert ein Intervall von Tagen. Dieses Tages-Intervall wird begrenzt
   * durch das erste Dia des Tages @p firstSlide und das letzte Dia des
   * Tages @p lastSlide.
   */
  struct DayInterval
  {
    unsigned int firstSlide;
    unsigned int lastSlide;
  }; // END DayInterval


protected: // attribures:

  /** String mit der Aufloesung dieses Ablaufplanes. */
  QString m_resolutionStr;

  /** Dateiname der (aufloesungsabhaengigen) Abspieldatei. */
  QString m_scheduleFilenameStr;

  /** Dateiname der Abspieldatei incl. Pfad. */
  QString m_filename;

  /** Name der Sound-Datei. */
  QString m_soundfilename;

  /** Menge der Dias dieses Abspielplanes. */
  vector<Slide> m_slides;

  /** Menge der Tage. */
  vector<DayInterval> m_days;


private:

  FILE         *pFile;
  unsigned int  m_slideIdx;  // 0 <= m_slideidx < #m_slides

};


#endif // SCHEDULE_H
