/***************************************************************************
    schedule.cpp  -  Ablaufplan einer Praesentation
    ------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.14 $
    CVS-ID    : $Id: schedule.cpp,v 1.14 2002/02/04 18:58:07 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Application include files:
#include "schedule.h"
#include "properties.h"


// erzeugt in main.cpp:
extern Properties *properties;


Schedule::Schedule(const QString& resolution,
                   const QString& schedulefilename) :
  m_resolutionStr(resolution),
  m_scheduleFilenameStr(schedulefilename)
{
  QString base = properties->base();

  m_filename = base + m_scheduleFilenameStr;

  pFile = fopen(m_filename.data(), "r");

  // Eine aufloesungsabhaengige Ablaufplan-Datei (z.B. fein.ply oder grob.ply)
  // ist folgendermassen aufgebaut:
  //
  // +----------------------------------------------------------------+
  // | 1: "Name der Sound-Datei"                                      |
  // | 2: "abs. Zeit in msec. fuer Dia-Start" "???" "Grafikdateiname" |
  // | 3: "abs. Zeit in msec. fuer Dia-Start" "???" "Grafikdateiname" |
  // |    ...                                                         |
  // | n: "abs. Zeit in msec. fuer Ende der Praes." "???" <<<ende>>>  |
  // +----------------------------------------------------------------+
  //
  // Die doppelten Anfuehrungszeichen sind nicht Teil der Zeichenketten
  // in der Datei, sie sollen die Grenzen der einzelnen Strings hier
  // markieren. Auch die Zeilennummern sind nicht Teil der Datei.
  // <<<ende>>> steht fuer die String-Konstante ende, die als solche das
  // Ende der Praesentation anzeigt.
  //
  // Am Ende dieser Datei zeigt ein Beispiel, wie aus Ablaufplan-Datei
  // Ein Ablaufplan der Klasse Schedule wird.

  if ( pFile == 0 )
    throw CanNotOpenFile(m_filename);

  // === erste Zeile enthaelt den Namen der Sound-Datei ===
  if ( !feof(pFile) )
  {
    m_soundfilename = base + getline(pFile);
  } // END if


  // === uebrige Zeilen repreasentieren die Dias ===

  QString line;
  while ( !feof(pFile) )
  {
    line = getline(pFile);

    if ( !feof(pFile) )
    {
      /*
       *   88888 99999 foobar
       *        ^     ^
       *      pos1   pos2
       */
      int pos1, pos2;
      pos1 = line.find(" ");
      pos2 = line.findRev(" ");

      QString absTimeStr  = line.mid(0, pos1);
      QString dummyStr    = line.mid(pos1+1, pos2);
      QString filenameStr = line.mid(pos2+1, line.length());

      // konvertiere backslash (DOS/Win) in slash (Unix):
      int bslashpos = filenameStr.find("\\");
      if ( bslashpos != -1 )
  	  {
        filenameStr[bslashpos] = '/';
      } // END if

  	  int abstime, dummy;
      sscanf(absTimeStr.data(), "%i", &abstime);
      sscanf(dummyStr.data(), "%i", &dummy);

      if ( filenameStr != "ende")
      {
        filenameStr = base + filenameStr;
      } // END if

      m_slides.push_back(Slide(abstime, dummy, filenameStr));

    } // END if !feof()

  } // END while

  // erste Zeile im Ablaufplan zeigt auf Default-Grafik (siehe Ende der Datei)
  m_slides.insert(m_slides.begin(),
                  Slide(0, 0, base + properties->defaultImageFilename()));

  // === fuehre Konvertuerung nach unten stehendem Schema durch ===
  for ( unsigned int idx=0; idx <(m_slides.size()-1); idx++ )
  {
    m_slides[idx].dummy( m_slides[idx+1].dummy() );
    m_slides[idx].duration( m_slides[idx+1].absTime() -
                            m_slides[idx].absTime()     );
  } // END for
  m_slides[m_slides.size()-1].dummy(0);
  m_slides[m_slides.size()-1].duration(0);

  // === Fuelle die 'Menge der Tage' ===

  DayInterval day;

  // erster 'Tag' ist die Default-Grafik
  day.firstSlide = 0;

  // ein neuer Tag beginnt mit einem Dia, dessen Dateiname die Zeichenkette
  // '00.' enthaelt (z.B. bild1200.jpg)
  for ( unsigned int idx = 1; idx < m_slides.size(); idx++ )
  {
    if ( (m_slides[idx].picFilename()).find("00.") != -1 )
    {
      // es hat schon einen Tag gegeben, der ein Dia vorher zuende ging
      day.lastSlide = idx-1;
      m_days.push_back(day);

      // ein neuer Tag beginnt nun
      day.firstSlide = idx;

    } // END if

  } // END for

  if ( day.firstSlide > day.lastSlide )
  {
    // der letzte Tag muss noch abgeschlossen werden
    day.lastSlide = m_slides.size()-1;
    m_days.push_back(day);
  } // END if


  // --- erstes Dia als aktuelles Dia ---
  m_slideIdx = 0;

  if ( pFile != 0 )
    fclose(pFile);

  pFile = 0;

} // END Schedule()


Schedule::~Schedule()
{
  while ( m_slides.size() > 0 )
    m_slides.erase(m_slides.begin());

} // END ~Schedule()


QString Schedule::resolutionString() const
{
  QString result = m_resolutionStr;
  
  int pos = m_scheduleFilenameStr.findRev(".");
  if ( pos != -1 )
  {
    result = result + " (" + m_scheduleFilenameStr.left(pos) + ")";
  } // END if
  
  return result;
} // END resolutionString()


QString& Schedule::scheduleFilename()
{
  return m_filename;
} // END scheduleFilename()


QString& Schedule::soundFilename()
{
  return m_soundfilename;
} // END soundFilename()


unsigned int Schedule::size() const
{
  return m_slides.size();
} // END size()


int Schedule::currSlideIdx() const
{
  return m_slideIdx;
} // END currSlideIdx()


int Schedule::duration() const
{
  return m_slides[m_slides.size()-1].absTime();
} // END duration()


Slide& Schedule::slide(unsigned int idx)
{
  if ( idx >= m_slides.size() )
    throw OutOfRange();

  return m_slides[idx];
} // END slide()


Slide& Schedule::slide()
{
  return m_slides[m_slideIdx];
} // END slide()


bool Schedule::nextSlide(Slide& nextslide)
{
  if ( m_slideIdx < m_slides.size() )
  {
    nextslide = m_slides[m_slideIdx+1];

    return true;
  } // END if

  return false;
} // END nextSlide()


void Schedule::prevDay(unsigned int picIdx, unsigned int& newPicIdx)
{
  unsigned int dayIdx = partOfDay(picIdx);

  if ( picIdx == m_days[dayIdx].firstSlide )
  {
    // picIdx ist erstes Dia des Tages, gehe zum vorigen Tag, wenn dieses
    // nicht der erste Tag ist (dann bleibe in diesem Tag)
    if ( dayIdx > 0 )
      newPicIdx = m_days[dayIdx-1].firstSlide;
    else
      newPicIdx = m_days[dayIdx].firstSlide;
  }
  else
  {
    // picIdx ist mitten im Tag, gehe zum Anfang des Tages
      newPicIdx = m_days[dayIdx].firstSlide;
  } // END if

} // END prevDay()


bool Schedule::nextDay(unsigned int picIdx, unsigned int& newPicIdx)
{
  unsigned int dayIdx = partOfDay(picIdx);

  // gehoert picIdx schon zum letzten Tag ?
  if ( dayIdx == m_days.size()-1 )
    return false;

  // sonst nimm erstes Dia vom naechsten Tag
  newPicIdx = m_days[dayIdx+1].firstSlide;

  return true;
} // END nextDay()


void Schedule::reset()
{
  m_slideIdx = 0;
} // END reset()


QString Schedule::getline(FILE *pFile)
{
  QString line;
  char buffer[255];

  if ( (pFile != 0) && (!feof(pFile)) )
  {
    // lese naechste Zeile:
    fgets(buffer, 255, pFile); // *** FIXME: buffer overflow moeglich !!!

    line = buffer;
    if ( line[line.length()-1] == '\n' )
    {
      line = line.mid(0, line.length()-2);
    } // END if

    return line;
  } // END if

  // ansonsten keine Zeile mehr uebrig
  return "";

} // END getline()


unsigned int Schedule::partOfDay(unsigned int slideIdx) const
{
  DayInterval day;

  for ( unsigned int dayIdx = 0; dayIdx < m_days.size(); dayIdx++ )
  {
    day = m_days[dayIdx];

    if ( (day.firstSlide <= slideIdx) && (slideIdx <= day.lastSlide) )
    {
#ifndef NDEBUG
      cerr << "Schedule::partOfDay(); Dia " << slideIdx
           << " geheorte zum Tag Nr. " << dayIdx << endl;
#endif

      return dayIdx;
    } // END if

  } // END for

  // bis hier darf man nicht kommen !
  return 0;
} // END partOfDay()


// Schema (Beispiel), nach dem aus der Abspiel-Datei (Original) der
// Ablaufplan wird (Konvertiert).
//
// Original:
// =========
//
//  +------------------------------------------+
//  | teil2.wav                                |
//  | 000:    4000 4000 1536x1024\bild1200.jpg |
//  | 001:   15000 4000 1536x1024\bild1201.jpg |
//  | 002:   28000 4000 1536x1024\bild1202.jpg |
//  | 003:   41000 4000 1536x1024\bild1203.jpg |
//  |      ...                                 |
//  | 196: 2469700 4000 1536x1024\bild2114.jpg |
//  | 197: 2483150 4000 1536x1024\bild2115.jpg |
//  | 198: 2496600 4000 1536x1024\bild2116.jpg |
//  | 199: 2510050 4000 1536x1024\bild2117.jpg |
//  | 200: 2521500 4000 ende                   |
//  +------------------------------------------+
//
//
// Konvertiert:
// ============
//
//  +------+---------+----------+-------+------------------------+
//  | Idx  | absTime | duration | dummy |            picFilename |
//  +------+---------+----------+-------+------------------------+
//  | 000: |       0 |     4000 |  4000 |            default.jpg |
//  | 001: |    4000 |    11000 |  4000 | 1536x1024\bild1200.jpg |
//  | 002: |   15000 |    13000 |  4000 | 1536x1024\bild1201.jpg |
//  | 003: |   28000 |    13000 |  4000 | 1536x1024\bild1202.jpg |
//  | 004: |   41000 |    11000 |  4000 | 1536x1024\bild1203.jpg |
//  |        ...                                                 |
//  | 197: | 2469700 |    13450 |  4000 | 1536x1024\bild2114.jpg |
//  | 198: | 2483150 |    13450 |  4000 | 1536x1024\bild2115.jpg |
//  | 199: | 2496600 |    13450 |  4000 | 1536x1024\bild2116.jpg |
//  | 200: | 2510050 |    11450 |  4000 | 1536x1024\bild2117.jpg |
//  | 201: | 2521500 |        0 |     0 | ende                   |
//  +------+---------+----------+-------+------------------------+
