/***************************************************************************
    ktourpresenterview.cpp  -  Haupt-Widget von KTourPresenter
    ----------------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.30 $
    CVS-ID    : $Id: ktourpresenterview.cpp,v 1.30 2002/02/14 16:38:17 dirk Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// C include files:
#include <math.h>
#include <stdio.h>

// Qt include files:
#include <qpainter.h>
#include <qfile.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcolor.h>
#include <qtooltip.h>

// KDE include files:
#include <kcursor.h>
#include <kstddirs.h>
#include <klocale.h>
#include <kimageio.h>

// Application include files:
#include "ktourpresenterview.h"
#include "ktourpresenterdoc.h"
#include "ktourpresenter.h"
#include "properties.h"


// erzeugt in main.cpp:
extern Properties *properties;


KTourPresenterView::KTourPresenterView(QWidget *parent, const char *name) : 
  QWidget(parent, name)
{
  setBackgroundMode(PaletteBase);

  setMinimumSize(320, 240);

  KImageIO::registerFormats();

  // Starte im Fensterdarstellungsmodus
  currMode = Window;

  /* Widgets:
     -------- */
  
  // pro Modus ein Image !!!
  images.push_back(new ImageWindow(this, "windowimage"));
  images.push_back(new ImageWindow(0L, "fullscreenimage",
      WStyle_Customize | WStyle_NoBorder | WStyle_StaysOnTop));
  images[Window]->setFullscreen(false);
  images[Fullscreen]->setFullscreen(true);
  images[Fullscreen]->setFixedSize(QApplication::desktop()->size());
  connect(images[Window], SIGNAL(sigPopup(const QPoint&)),
          topLevelWidget(), SLOT(slotShowPopup(const QPoint&)));
  connect(images[Fullscreen], SIGNAL(sigPopup(const QPoint&)),
          topLevelWidget(), SLOT(slotShowPopup(const QPoint&)));
  images[Fullscreen]->hide(); // erstmal kein Vollbildmodus

  // Fortschrittszeile:
  progressLabel = new QLabel(i18n("Progress: "), this);
  slider        = new ProgressSlider(this, "progress");
  actTime       = new QLabel("00:00", this, "acttime");
  slashLabel    = new QLabel("/", this, "slash");
  totalTime     = new QLabel("00:00", this, "totaltime");
  connect(slider, SIGNAL(sliderReleased()), this, SLOT(slotNewValue()));
  // *** FIXME: vvv statt ^^^; klappt bisher noch nicht ! Beim Diawechsel
  //                           kommt es zu einer Art Hall
  //connect(slider, SIGNAL(valueChanged(int)), this, SLOT(slotNewValue(int)));

  /* Layout:
     ------- */
  
  QVBoxLayout *topLayout = new QVBoxLayout(this, 0, -1, "toplayout");
  QHBoxLayout *progressLayout = new QHBoxLayout(0, 0, -1, "progresslayout");
    
  topLayout->addWidget(images[Window]);
  topLayout->addLayout(progressLayout);
  
  progressLayout->addWidget(progressLabel);
  progressLayout->addWidget(slider);
  progressLayout->addWidget(new QLabel(" ", this));
  progressLayout->addWidget(actTime);
  progressLayout->addWidget(slashLabel);
  progressLayout->addWidget(totalTime);
  progressLayout->addWidget(new QLabel(" ", this));

  QToolTip::add(actTime, i18n("Actual time since begining of the "
                                "presentation."));
  QToolTip::add(totalTime, i18n("Total time of the presentation."));

  setProgressEnabled(false);
} // END KTourPresenterView()


KTourPresenterView::~KTourPresenterView()
{
} // END ~KTourPresenterView()


KTourPresenterDoc *KTourPresenterView::getDocument() const
{
  KTourPresenterApp *theApp = (KTourPresenterApp *) parentWidget();

  return theApp->getDocument();
} // END getDocument()


#if KDE_VERSION >= 220
void KTourPresenterView::setAutoHideEnabled(bool flag)
{
  images[Window]->setAutoHideEnabled(flag);
  images[Fullscreen]->setAutoHideEnabled(flag);
} // END setAutoHideEnabled()
#endif


void KTourPresenterView::loadImage(const QString& filename)
{
  currImageFilename = filename;

  images[currMode]->showNextImage(filename);
} // END loadImage


bool KTourPresenterView::cacheImage(const QString& filename)
{
  return images[currMode]->cacheImage(filename);
} // END cacheImage()


void KTourPresenterView::resizeEvent(QResizeEvent *re)
{
  QSize newsize = re->size();

  int viewWidth  = newsize.width();
  int viewHeight = newsize.height() - progressHeight();

  images[currMode]->resizeOptimal(viewWidth, viewHeight);

  // *** FIXME: ugly hack :-(
  if ( currMode == Window )
    images[currMode]->setFixedSize(QSize(viewWidth, viewHeight));
} // END resizeEvent()


void KTourPresenterView::eraseImage()
{
  // *** FIXME: loesche das Image mit Hintergrundfarbe

  // ...

} // END eraseImage()


void KTourPresenterView::showImage()
{
  // zeige die Pixmap:
  repaint();
} // END showImage()


void KTourPresenterView::setProgressEnabled(bool flag)
{
  progressLabel->setEnabled(flag);
  slider->setEnabled(flag);
  slashLabel->setEnabled(flag);
  actTime->setEnabled(flag);
  totalTime->setEnabled(flag);
} // END setProgressEnabled()


int KTourPresenterView::progressHeight()
{
  int sliderHeight = slider->height();
  int labelHeight  = actTime->height();
  
  return (sliderHeight > labelHeight) ? sliderHeight : labelHeight;
} // END progressHeight()


void KTourPresenterView::setSliderMax(int maxValue)
{
  slider->setMaxValue(maxValue);
} // END setSliderMax()


void KTourPresenterView::updateTotalTime(int msec)
{
  QString totalStr = msecToString(msec);  
  totalTime->setText(totalStr);
} // END updateTotalTime()


void KTourPresenterView::updateActTime(int msec)
{
  QString totalStr = msecToString(msec);  
  actTime->setText(totalStr);

  slider->setValue(msec);
} // END updateActTime()


void KTourPresenterView::updateSlider(int msec)
{
  slider->setValue(msec);
} // END updateSlider()


KTourPresenterView::ViewMode KTourPresenterView::viewMode() const
{
  return currMode;
} // END viewMode()


void KTourPresenterView::slotNewValue()
{
  emit sigNewSliderPos(slider->getSliderPos());
} // END slotNewValue()


void KTourPresenterView::slotNewValue(int value)
{
  emit sigNewSliderPos(value);
} // END slotNewValue()


void KTourPresenterView::slotReset()
{
  actTime->setText("00:00");
  totalTime->setText("00:00");

  if ( currMode == Fullscreen )
    slotToggleFullscreen();

  // loesche Image(s)
  images[Fullscreen]->eraseImage();
  images[Window]->eraseImage();

  setProgressEnabled(false);
} // END slotReset()


void KTourPresenterView::slotBackgroundColorChanged(const QColor& color)
{
  images[Window]->setBackgroundColor(color);
  images[Fullscreen]->setBackgroundColor(color);
} // END slotBackgroundColorChanged()


void KTourPresenterView::slotModifierChanged()
{
  // teile dem Image mit, dass sich Modifier geaendert haben !!!
  images[currMode]->setBrightness(properties->brightness());
  images[currMode]->setContrast(properties->contrast());
  images[currMode]->setGamma(properties->gamma());
} // END slotModifierChanged()


void KTourPresenterView::slotMoreBrightness()
{
  images[currMode]->slotMoreBrightness();
} // END slotMoreBrightness()


void KTourPresenterView::slotLessBrightness()
{
  images[currMode]->slotLessBrightness();
} // END slotLessBrightness()


void KTourPresenterView::slotMoreContrast()
{
  images[currMode]->slotMoreContrast();
} // END slotMoreContrast()


void KTourPresenterView::slotLessContrast()
{
  images[currMode]->slotLessContrast();
} // END slotLessContrast()


void KTourPresenterView::slotMoreGamma()
{
  images[currMode]->slotMoreGamma();
} // END slotMoreGamma()


void KTourPresenterView::slotLessGamma()
{
  images[currMode]->slotLessGamma();
} // END slotLessGamma()


void KTourPresenterView::slotResetModifier()
{
  properties->brightness(0);
  properties->contrast(0);
  properties->gamma(0);

  // *** FIXME: signalisiere, dass sich die Modifier geaendert haben !
  images[currMode]->setBrightness(0);
  images[currMode]->setContrast(0);
  images[currMode]->setGamma(0);
} // END slotResetModifier()


void KTourPresenterView::slotToggleFullscreen()
{
  ViewMode oldMode;

  if ( currMode == Window )
  {
    // nun Vollbildmodus:
    currMode = Fullscreen;
    oldMode = Window;

    images[Fullscreen]->show();
    images[Window]->eraseImage();

    images[Fullscreen]->setFocus();
  }
  else
  {
    // nun Anwendungsfenstermodus:
    currMode = Window;
    oldMode = Fullscreen;

    images[Fullscreen]->hide();
    images[Fullscreen]->eraseImage();

    images[Window]->setFocus();
  } // END if

  // setze Modifier im neuen Modus
  images[currMode]->setBrightness(properties->brightness());
  images[currMode]->setContrast(  properties->contrast());
  images[currMode]->setGamma(     properties->gamma());

  // lade aktuelle Grafik fuer neuen Modus:
  images[currMode]->showNextImage(currImageFilename);
} // END slotToggleFullscreen()


QString KTourPresenterView::msecToString(int msec)
{
  double dsecs = (double) msec / 1000.0;
  int     secs = (int) rint(dsecs);
  
  int minutes = secs / 60;
  int seconds = secs % 60;

  QString minStr = "";
  minStr.sprintf("%i", minutes);
  if ( minutes < 10 )
    minStr = "0" + minStr;
  
  QString secStr = "";
  secStr.sprintf("%i", seconds);
  if ( seconds < 10 )
    secStr = "0" + secStr;
  
  return minStr+":"+secStr;
} // END msecToString()


inline ImageWindow *KTourPresenterView::image(ViewMode mode)
{
  int idx = ( (mode == Window) ? 0 : 1);

  return images[idx];
} // END image()
