/***************************************************************************
    imlibwidget.cpp  -  Bildet ein Imlib-Image auf ein Qt Widget ab
    ---------------
    copyright : (C) 2001, 2002 by Dirk Rosert
    email     : dirk@kiste.ping.de
    author    : $Author: dirk $
    revision  : $Revision: 1.20 $
    CVS-ID    : $Id: imlibwidget.cpp,v 1.20 2002/02/02 21:48:24 dirk Exp $

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * basiert auf KuickShow 0.8.2 von Carsten Pfeiffer <pfeiffer@kde.org>
 */


// C include files:
#include <sys/time.h>
#include <unistd.h>

// STL include files:
#include <iostream>
#include <cstdlib>

// Qt include files:
#include <qcolor.h>
#include <qfile.h>
#include <qobject.h>
#include <qpalette.h>
#include <qwidget.h>

// KDE include files:
#include <kcursor.h>

// Application include files:
#include "imlibwidget.h"
#include "properties.h"


// erzeugt in main.cpp:
extern Properties *properties;


ImlibWidget::ImlibWidget(QWidget *parent, const char *name, WFlags f) :
  QWidget(parent, name, f)
{
  deleteImlibData = true;

  ImlibInitParams par;

  // PARAMS_PALETTEOVERRIDE taken out because of segfault in imlib :o(
  par.flags = ( PARAMS_REMAP |
      PARAMS_FASTRENDER | PARAMS_HIQUALITY | PARAMS_DITHER |
      PARAMS_IMAGECACHESIZE | PARAMS_PIXMAPCACHESIZE );

  par.paletteoverride   = properties->ownPalette()  ? 1 : 0;
  par.remap             = properties->fastRender()  ? 1 : 0;
  par.hiquality         = properties->dither16bit() ? 1 : 0;
  par.dither            = properties->dither8bit()  ? 1 : 0;
  unsigned int maxcache = properties->maxCache();

  // 0 == kein cache
  par.imagecachesize  = maxcache * 1024;
  par.pixmapcachesize = maxcache * 1024;

  id = Imlib_init_with_params( x11Display(), &par );

  init();

} // END ImlibWidget()


ImlibWidget::ImlibWidget(ImlibData *_id, QWidget *parent,
                         const char *name, WFlags f) :
  QWidget(parent, name, f)
{
  id              = _id;
  deleteImlibData = false;

  init();

} // END ImlibWidget()


ImlibWidget::~ImlibWidget()
{
  if ( deleteImlibData )
    delete id;

  if ( win )
    XDestroyWindow(x11Display(), win);

  delete imageCache;

} // END ~ImlibWidget()


void ImlibWidget::init()
{
  int w = 1; // > 0 fuer XCreateWindow
  int h = 1;
  m_backgroundColor = properties->backgroundColor();
  ktimg = 0L;

  if ( !id )
    fatal("ImlibWidget: Imlib not initialized, aborting.");

  setAutoRender(true);

  slotBackgroundColor(m_backgroundColor);

  // erzeuge Cache fuer (willkuerlich) drei Images:
  imageCache = new ImageCache(id, 3);

  connect(imageCache, SIGNAL(sigBusy()), SLOT(setBusyCursor()));
  connect(imageCache, SIGNAL(sigIdle()), SLOT(restoreCursor()));

  win = XCreateSimpleWindow(x11Display(), winId(), 0, 0, w, h, 0, 0, 0);

} // END init()


KTourImage *ImlibWidget::loadImageInternal(const QString& filename,
                                           bool paintFlag)
{
  // weise Standardmodifikatoren der Image zu:
  mod.brightness = ImlibOffset + properties->brightness();
  mod.contrast   = ImlibOffset + properties->contrast();
  mod.gamma      = ImlibOffset + properties->gamma();

  KTourImage *m_ktimg = imageCache->getKTimage(filename, mod, paintFlag);
  if ( !m_ktimg )
  {
    // Datei konnte nicht geladen werden, defekt oder falsches Format
    qDebug("  ImlibWidget: can't load image %s", filename.latin1());
    return 0L;
  } // END if

  // *** FIXME: vvv ist das noetig ???
  m_ktimg->renderPixmap();

  return m_ktimg;

} // END loadImageInternal()


bool ImlibWidget::loadImage(const QString& filename, bool paintFlag)
{
  KTourImage *m_ktimg = loadImageInternal(filename, paintFlag);

  if ( m_ktimg )
  {
    ktimg = m_ktimg;

    autoUpdate(true);
    m_filename = filename;
  }

  return ( m_ktimg != 0L );
} // END loadImage()


void ImlibWidget::eraseImage()
{
  ktimg = 0L;

  setAutoRender(true);
  slotBackgroundColor(properties->backgroundColor());

  if ( win )
    XDestroyWindow(x11Display(), win);

  //Imlib_destroy_image(id, ktimg->imlibImage());
  win = XCreateSimpleWindow(x11Display(), winId(), 0, 0, 1, 1, 0, 0, 0);

  showImage();
} // END eraseImage()


bool ImlibWidget::cacheImage(const QString& filename)
{
  KTourImage *myKtimg = loadImageInternal(filename);
  
  return ( myKtimg != 0L );
} // END cacheImage()


void ImlibWidget::showImage()
{
  XMapWindow(x11Display(), win);
  XSync(x11Display(), false);
} // END showImage()


void ImlibWidget::zoomImage(float factor)
{
  if ( factor == 1 || factor == 0 || !ktimg )
    return;

  float wf, hf;

  wf = (float) ktimg->width() * factor;
  hf = (float) ktimg->height() * factor;

  // minimale Groesse fuer eine Image sind 2x2 pixels
  if ( wf <= 2.0 || hf <= 2.0 )
    return;

  ktimg->resize((int) wf, (int) hf);
  autoUpdate(true);

} // END zoomImage()


// -256..256
void ImlibWidget::setBrightness(int factor)
{
    mod.brightness = factor + ImlibOffset;
    setImageModifier();

    properties->brightness(factor);

    autoUpdate();
} // END setBrightness()


// -256..256
void ImlibWidget::setContrast(int factor)
{
    mod.contrast = factor + ImlibOffset;
    setImageModifier();

    properties->contrast(factor);
    autoUpdate();
} // END setContrast()


// -256..256
void ImlibWidget::setGamma(int factor)
{
    mod.gamma = factor + ImlibOffset;
    setImageModifier();

    properties->gamma(factor);

    autoUpdate();
} // END setGamma()



void ImlibWidget::showImageOriginalSize()
{
  if ( !ktimg )
    return;

  ktimg->restoreOriginalSize();
  autoUpdate( true );

  showImage();
} // END showImageOriginalSize()


void ImlibWidget::slotBackgroundColor(const QColor& color)
{
  setPalette(QPalette(color));
  setBackgroundMode(PaletteBackground);
} // END slotBackgroundColor()


void ImlibWidget::updateWidget(bool geometryUpdate)
{
  if ( !ktimg )
    return;

  XSetWindowBackgroundPixmap(x11Display(), win, ktimg->pixmap());
  XClearWindow(x11Display(), win);

  if ( geometryUpdate )
    updateGeometry(ktimg->width(), ktimg->height());

  showImage();
} // END updateWidget()


// here we just use the size of ktimg, may be overridden in subclass
void ImlibWidget::updateGeometry(int w, int h)
{
  XMoveWindow(x11Display(), win, 0, 0);
  XResizeWindow(x11Display(), win, w, h);
  resize(w, h);
} // END updateGeometry()


void ImlibWidget::closeEvent(QCloseEvent *e)
{
  e->accept();

  QWidget::closeEvent(e);
} // END closeEvent()


void ImlibWidget::setBackgroundColor(const QColor& color)
{
  m_backgroundColor = color;
  setPalette(QPalette(m_backgroundColor));
} // END setBackgroundColor()


const QColor& ImlibWidget::backgroundColor() const
{
  return m_backgroundColor;
} // END backgroundColor


void ImlibWidget::setImageModifier()
{
  if ( !ktimg )
    return;

  Imlib_set_image_modifier(id, ktimg->imlibImage(), &mod);
  ktimg->setDirty(true); // laesst Grafik neuzeichnen
} // END setImageModifier()


int ImlibWidget::imageWidth() const
{
  return ktimg ? ktimg->width() : 0;
} // END imageWidth()


int ImlibWidget::imageHeight() const
{
  return ktimg ? ktimg->height() : 0;
} // END imageHeight()


void ImlibWidget::setAutoRender(bool enable)
{
  m_isAutoRendering = enable;
} // END setAutoRender()


bool ImlibWidget::isAutoRenderEnabled() const
{
  return m_isAutoRendering;
} // END isAutoRenderEnabled()


void ImlibWidget::setBusyCursor()
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::waitCursor());
} // END setBusyCursor()


void ImlibWidget::restoreCursor()
{
  if ( !properties->hideMouse() )
    setCursor(KCursor::arrowCursor());
} // END restoreCursor()
